"""
Stackless and Twisted Threaded Example
Copyright (C) 2007 David Wyand

AIObject class adds a thinking loop and steering
behavior to the MovingObject class.
"""

from movingobject import MovingObject
import objectdb
import uthread
import random
import time
import vector
import tasklet

class AIObject(MovingObject):
    """Uses 2D steering behaviors to follow a target"""
    def __init__(self, name="", channel=None):
        super(AIObject, self).__init__(name, channel)
        self.thinkHandler = self.__Think
        self.steeringBehaviorHandler = self.__SteeringBehavior
        self.updateHandler = self.__UpdateAIObject
        self.updateAI = False
        self.__steeringForce = [0,0]
    
    # Protected Methods
    
    def _Register(self, id):
        super(AIObject, self)._Register(id)
        self.updateAI = True
        tasklet.newObjTasklet(self, "AIObject::ThinkHandler", self.thinkHandler).run()

    def _Unregister(self):
        super(AIObject, self)._Unregister()
        self.updateAI = False
    
    # Private Methods

    def __SteeringBehavior(self, target):
        diff = [target.Position[0] - self.Position[0], target.Position[1] - self.Position[1]]
        vector.normalize(diff)
        desiredvel = [diff[0] * self.MaxSpeed, diff[1] * self.MaxSpeed]
        
        force = [desiredvel[0] - self.Velocity[0], desiredvel[1] - self.Velocity[1]]
        if vector.magnitude(force) > self.MaxForce:
            vector.normalize(force)
            force[0] *= self.MaxForce
            force[1] *= self.MaxForce
        
        return force
    
    def __Think(self):
        while self.updateAI:
            target = objectdb.FindObjectByName("Target")
            if target is not None:
                # Calculate the combined force from each steering behavior.
                self.__steeringForce = self.steeringBehaviorHandler(target)
            
            uthread.Sleep(0.5)
    
    def __UpdateAIObject(self):
        lastTime = time.clock()
        while self.updatePos:
            currentTime = time.clock()
            dt = currentTime - lastTime
            lastTime = currentTime
        
            # Acceleration = Force/Mass
            acceleration = [self.__steeringForce[0] / self.Mass, self.__steeringForce[1] / self.Mass]
        
            # Update velocity
            self.Velocity[0] += acceleration[0] * dt
            self.Velocity[1] += acceleration[1] * dt
        
            # Make sure object does not exceed maximum velocity
            if vector.magnitude(self.Velocity) > self.MaxSpeed:
                vector.normalize(self.Velocity)
                self.Velocity[0] *= self.MaxSpeed
                self.Velocity[1] *= self.MaxSpeed
            
            # Update position based on velocity
            self.Position[0] += self.Velocity[0] * dt
            self.Position[1] += self.Velocity[1] * dt

            # Update the heading if non-zero velocity
            if(vector.magnitude(self.Velocity) > 0.0000001):
                self.Heading = vector.normalized(self.Velocity)

            uthread.BeNice()
