"""
Stackless and Twisted Threaded Example
Copyright (C) 2007 David Wyand

Sets up the PYGame display to draw the space ships
and their target.  A seconds per frame HUD is also
provided.
"""

from objectbase import ObjectBase
from target import Target
from aiobject import AIObject
import objectdb
import uthread
import time
import pygame
import tasklet

class Display(ObjectBase):
    """Zone and object display including Seconds Per Frame using PYGame"""
    def __init__(self, channel=None):
        super(Display, self).__init__("Display", channel)
        self.processMsgHandler = self.__MsgHandler
        self.displayHandler = self.__UpdateDisplay
        self.updateDisplay = False
        
        pygame.init()
        self.window = None
        self.font = pygame.font.Font(pygame.font.get_default_font(), 12)

    # Protected Methods
    
    def _Unregister(self):
        super(Display, self)._Unregister()
        self.updateDisplay = False
        pygame.display.quit()

    # Private Methods
    
    def __MsgHandler(self, msg):
        (msgType, msgArgs) = msg[0], msg[1:]
        
        print "Display received:", str(msg)
        
        if msgType == "OPENWINDOW":
            if self.updateDisplay == False:
                self.updateDisplay = True
                tasklet.newObjTasklet(self, "Display::DisplayHandler", self.displayHandler).run()
            
        elif msgType == "CLOSEWINDOW":
            self.updateDisplay = False

    def __UpdateDisplay(self):
        lastTime = time.clock()
        
        # Open the window
        pygame.display.init()
        size = (500, 500)
        zone = objectdb.FindObjectByName("Zone")
        if zone is not None:
            size = (zone.Size[0], zone.Size[1])
    
        self.window = pygame.display.set_mode(size)
        pygame.display.set_caption("Stackless Twisted Example by David Wyand")

        # Seconds Per Frame sliding window -- updated per frame
        spfList = [0] * 90
        
        # Update the display
        while self.updateDisplay:
            currentTime = time.clock()
            dt = currentTime - lastTime
            lastTime = currentTime
            
            for event in pygame.event.get():
                if event.type == pygame.QUIT:
                    self.updateDisplay = False
                if event.type == pygame.KEYDOWN and event.key == pygame.K_ESCAPE:
                    self.updateDisplay = False

            if self.updateDisplay == False:
                break
            
            screen = pygame.display.get_surface()

            # Clear the background
            background = pygame.Surface(screen.get_size())
            background = background.convert()
            background.fill((200, 200, 200))
            screen.blit(background, (0,0))

            # Track seconds per frame stats on sliding window
            spfList.pop(0)
            spfList.append(dt)
            spfmin = min(spfList)
            spfmax = max(spfList)
            spfavg = 0
            for spf in spfList:
                spfavg += spf
            spfavg /= len(spfList)
            
            # Draw the seconds per frame stats
            text = "SPF Avg: %.4f" % spfavg
            surf = self.font.render(text, True, (0,0,0))
            screen.blit(surf, (2, 2))
            text = "Max: %.4f" % spfmax
            surf = self.font.render(text, True, (0,0,0))
            screen.blit(surf, (132, 2))
            text = "Min: %.4f" % spfmin
            surf = self.font.render(text, True, (0,0,0))
            screen.blit(surf, (232, 2))

            # Draw the objects
            for o in objectdb.GetEnumerator():
                if(isinstance(o, AIObject)):
                    red = 64 + int((o.MaxSpeed - o.maxspeedmin) / float((o.maxspeedmax - o.maxspeedmin)) * (255 - 64))
                    pygame.draw.circle(screen, (red, 64, 64), (o.Position[0], o.Position[1]), 3)
                    pygame.draw.aaline(screen, (192, 128, 0), (o.Position[0], o.Position[1]), (o.Heading[0]*10+o.Position[0],o.Heading[1]*10+o.Position[1]))
                    
                elif(isinstance(o, Target)):
                    linelen = 6
                    pygame.draw.aaline(screen, (64, 64, 64), (o.Position[0], o.Position[1] - linelen), (o.Position[0], o.Position[1] + linelen))
                    pygame.draw.aaline(screen, (64, 64, 64), (o.Position[0] - linelen, o.Position[1]), (o.Position[0] + linelen, o.Position[1]))

            pygame.display.flip()

            uthread.BeNice()

        # Close the window
        pygame.display.quit()
