"""
Stackless and Twisted Threaded Example
Copyright (C) 2007 David Wyand

This application runs Stackless as a separate Twisted thread
to demonstrate their coexistence and communication.  The
Stackless code creates a number of AI space ships that fly
around a 2D zone chasing after a target.  Each ship has its
own flight characteristics.  The Twisted code provides a web
back end to the simulation.

When the application is first started a PYGame window is
opened displaying the space ships and the target crosshairs.
Each ship is drawn as a red circle with the shade of red
indicating the ship's relative maximum speed.  An orange line
coming from the center of the circle represents the ship's
heading.

At the top of the window are the 'seconds per frame' statistics
as calculated using a sliding time window.  If the PYGame
window is closed the simulation will continue to run but without
spending time on drawing the display.

The Twisted side of the application provides another view into
the simulation in the form of a web page.  To access the page
locally use:

http://127.0.0.1:8001

The top of the page has three buttons.  The 'Quit' button
will shutdown the simulation and the PYGame window if it is
open.  The 'Open Display' button will open the PYGame window
if it was closed.  The last button allows the user to
toggle between a list of ships and a list of tasklets.

The first web page displays the position of the target and each
ship at the time the page was loaded.  Each ship's simulation
ID and name is also provided.

The second web page provides a list of all Stackless
tasklets as obtained from the garbage collector.  It lists
who created the tasklet, what context the tasklet was created
under, the tasklet's class, and some additional information
that is available from Stackless.  The 'main' tasklet is
shaded in gray while the current tasklet is shaded in yellow.

Tested with PYGame, Stackless and Twisted for Python 2.4
"""

from twisted.internet import reactor
import stacklessthread
import stackless
import webpages
import comm

if __name__ == "__main__":
    # Set up the Twisted to Stackless communications channels
    comm.SetupCommChannels()
    
    # Start Stackless work in another thread
    reactor.callInThread(stacklessthread.run, comm.GetCommChannel(), comm.GetDisplayChannel())
    
    # Set up the web pages
    webpages.Setup()
    
    reactor.run()
