"""
Stackless and Twisted Threaded Example
Copyright (C) 2007 David Wyand

MovingObject class adds 2D motion to the ObjectBase
class.
"""

from objectbase import ObjectBase
import objectdb
import uthread
import random
import time
import vector
import tasklet

class MovingObject(ObjectBase):
    """An object that has a position and velocity in 2D space"""
    def __init__(self, name="", channel=None):
        super(MovingObject, self).__init__(name, channel)
        self.updateHandler = self.__UpdateMovingObject
        self.updatePos = False
        
        zone = objectdb.FindObjectByName("Zone")
        if zone is not None:
            self.__pos = [random.randint(0, zone.Size[0]), random.randint(0, zone.Size[0])]
        else:
            self.__pos = [0, 0]
        
        self.__velocity = [0, 0]
        self.__heading = [0, 1]
        self.__mass = 1.0
        
        self.maxspeedmin = 20
        self.maxspeedmax = 60
        self.__maxspeed = random.randint(self.maxspeedmin, self.maxspeedmax)
        
        self.__maxforce = random.randint(10, 20)
    
    # Properties
    
    def __GetPos(self):
        return self.__pos
    
    Position = property(__GetPos)
    
    def __GetVelocity(self):
        return self.__velocity
    
    def __SetVelocity(self, vel):
        self.__velocity = vel
    
    Velocity = property(__GetVelocity, __SetVelocity)
    
    def __GetHeading(self):
        return self.__heading
    
    def __SetHeading(self, heading):
        self.__heading = heading
    
    Heading = property(__GetHeading, __SetHeading)
    
    def __GetMass(self):
        return self.__mass
    
    def __SetMass(self, mass):
        self.__mass = mass
    
    Mass = property(__GetMass, __SetMass)
    
    def __GetMaxSpeed(self):
        return self.__maxspeed
    
    def __SetMaxSpeed(self, speed):
        self.__maxspeed = speed
    
    MaxSpeed = property(__GetMaxSpeed, __SetMaxSpeed)
    
    def __GetMaxForce(self):
        return self.__maxforce
    
    def __SetMaxForce(self, force):
        self.__maxforce = force
    
    MaxForce = property(__GetMaxForce, __SetMaxForce)
    
    # Protected Methods
    
    def _Register(self, id):
        super(MovingObject, self)._Register(id)
        self.updatePos = True
        tasklet.newObjTasklet(self, "MovingObject::UpdateHandler", self.updateHandler).run()

    def _Unregister(self):
        super(MovingObject, self)._Unregister()
        self.updatePos = False
    
    # Private Methods
    
    def __UpdateMovingObject(self):
        lastTime = time.clock()
        while self.updatePos:
            currentTime = time.clock()
            dt = currentTime - lastTime
            lastTime = currentTime
            
            # Update position based on velocity
            self.Position[0] += self.Velocity[0] * dt
            self.Position[1] += self.Velocity[1] * dt

            # Update the heading if non-zero velocity
            if(vector.magnitude(self.Velocity) > 0.0000001):
                self.Heading = vector.normalized(self.Velocity)

            uthread.BeNice()
