"""
Stackless and Twisted Threaded Example
Copyright (C) 2007 David Wyand

Defines the area in which the space ships and target
move and provides simulation communications with
Twisted.
"""

from objectbase import ObjectBase
from aiobject import AIObject
from target import Target
import objectdb
import uthread
import tasklet
import stackless

class Zone(ObjectBase):
    """The zone in which all objects move.  Also provides basic communications with Twisted."""
    def __init__(self, channel=None):
        super(Zone, self).__init__("Zone", channel)
        self.processMsgHandler = self.__MsgHandler
        self.__size = [500, 500]
        self.__seekerNum = 1
    
    # Properties
    
    def __GetSize(self):
        return self.__size
    
    Size = property(__GetSize)

    # Public Methods
    
    def AddSeeker(self):
        obj = AIObject()
        obj.Name = "Seeker" + "%03d" % self.__seekerNum
        objectdb.Register(obj)
        
        self.__seekerNum += 1
    
    def RemoveSeeker(self, id):
        obj = objectdb.FindObject(id)
        if obj is not None:
            objectdb.Unregister(id)
        
    # Private Methods
    
    def __StartupZone(self):
        obj = Target()
        objectdb.Register(obj)
        uthread.Sleep(1)
        
        for i in range(50):
            self.AddSeeker()
            uthread.Sleep(0.25)
    
    def __MsgHandler(self, msg):
        (msgType, msgArgs) = msg[0], msg[1:]
        
        #print "Zone received:", msg
        
        if msgType == "STARTUP":
            tasklet.newObjTasklet(self, "Zone::StartupZone", self.__StartupZone).run()
        
        elif msgType == "SHUTDOWN":
            import stacklessthread
            stacklessthread.Running = False
        
        elif msgType == "ADDSEEKER":
            self.AddSeeker()
        
        elif msgType == "REMOVESEEKER":
            self.RemoveSeeker(msgArgs)
        
        elif msgType == "GETAIOBJECTS":
            tasklet.newObjTasklet(self, "Zone::GetAIObjectsForWeb", self.GetAIObjectsForWeb, msgArgs).run()
        
        elif msgType == "GETTASKLETS":
            tasklet.newObjTasklet(self, "Zone::GetTaskletsForWeb", self.GetTaskletsForWeb, msgArgs).run()
    
    def GetAIObjectsForWeb(self, args):
        from twisted.internet import reactor
        import webpages
        
        tpos = ""
        target = objectdb.FindObjectByName("Target")
        if target is not None:
            tpos = target.Position
        
        db = {}
        for o in objectdb.GetEnumerator():
            if isinstance(o, AIObject):
                db[o.ID] = (o.Name, o.Position)
            #uthread.BeNice()
        
        (page, request) = args
        reactor.callFromThread(page.ReceiveAIObjectList, tpos, db, request)
    
    def GetTaskletsForWeb(self, args):
        from twisted.internet import reactor
        import webpages
        import gc

        taskletList = []
        gc.collect()
        oo = gc.get_objects()
        for o in oo:
            if getattr(o, "__class__", None):
                if isinstance(o, stackless.tasklet):
                    classModule = o.__class__.__module__ + "." + o.__class__.__name__
                    
                    try:
                        context = o.context
                    except AttributeError:
                        context = ""
                    
                    try:
                        name = o.parentObj.Name
                    except AttributeError:
                        name = ""
                    
                    taskletList.append((o.thread_id, name, context, classModule, id(o), o.blocked, o.is_current, o.is_main))
        
        (page, request) = args
        reactor.callFromThread(page.ReceiveTaskletList, taskletList, request)
