/*
**	Revision History:
**	February 11, 2002	David Wyand		Began changes to convert for LW use (was MilkShape)
**	February 28, 2002	David Wyand		Corrected the generation of polygon normals.
**	March 7, 2002		David Wyand		Corrected the v (as in uv) coordinate for textures
**	March 13, 2002		David Wyand		Added mesh transform into world coordinates to take into
**										account an objected being moved in Layout from the origin.
**										The time at which this transform is taken is based on
**										instdata->m_fBaseTransFrame
*/

#pragma warning ( disable: 4786 )
#include <vector>
#include <map>
#include <cmath>

#include "LWGlobal.h"

#include "DTSLWMesh.h"
#include "LWObjectDB.h"

namespace DTS
{

	// -----------------------------------------------------------------------
	//  Imports a Lighwave Mesh
	// -----------------------------------------------------------------------

	CLWMesh::CLWMesh (CLWData* instdata, LWItemID itemID, int rootBone) : Mesh (Mesh::T_Standard)
	{
		int v, n ;
		int nPPType = instdata->m_nPolyPointType;
		LWDVector	vWorldPos, vWorldPiv;
		LWDVector	vWorldRSX, vWorldRSY, vWorldRSZ;

		//*** Create a default material index
		int materialIndex = 0;

		//*** Build the database for this object
		CLWObjectDB odb( itemID, CLWGlobal::global );
		if ( !(odb.isLoaded()) ) {
			//*** TODO: Need to handle this error better -- Will CRASH LightWave as it stands.
			CLWGlobal::msg->error( "Couldn't allocate object database for", CLWGlobal::iteminfo->name( itemID ));
			return;
		}

		//*** Get the world position and rotation/scale of this object
		CLWGlobal::iteminfo->param(itemID, LWIP_W_POSITION, instdata->m_fBaseTransFrame, vWorldPos);
		CLWGlobal::iteminfo->param(itemID, LWIP_RIGHT, instdata->m_fBaseTransFrame, vWorldRSX);
		CLWGlobal::iteminfo->param(itemID, LWIP_UP, instdata->m_fBaseTransFrame, vWorldRSY);
		CLWGlobal::iteminfo->param(itemID, LWIP_FORWARD, instdata->m_fBaseTransFrame, vWorldRSZ);
		CLWGlobal::iteminfo->param(itemID, LWIP_PIVOT, instdata->m_fBaseTransFrame, vWorldPiv);

		//*** Import vertex position and texture data
		for (v = 0 ; v < odb.m_nPoints ; v++)
		{
			float lvector[3];	//*** The coordinates of a single point locally
			float vector[3];	//*** The coordinates of a single point after LW world transform

			//*** Obtain the coordinates of a single point.  The position is in the
			//*** form of pos[a][b] where a=0 for the initial position and
			//*** a=1 for the transformed position at the current frame;
			//*** b=0..2 for the x,y,z of the point.
			//*** NOTE: For this to work, the bounding box parameters in Layout must
			//*** be set so that all points for the current frame are displayed.
			//*** (Perhaps only required when obtaining trandformed position??)
			lvector[0] = (odb.m_pPoints[ v ].pos[ nPPType ][ 0 ]);
			lvector[1] = (odb.m_pPoints[ v ].pos[ nPPType ][ 1 ]);
			lvector[2] = (odb.m_pPoints[ v ].pos[ nPPType ][ 2 ]);

			//*** If this object is attached to the Root node, do world trasform,
			//*** rotation, and scale
//			if(rootBone == instdata->m_nRootNode)
//			{
				int i;
				for(i=0; i<3; ++i)
				{
					vector[i] = (lvector[0] - vWorldPiv[0])*vWorldRSX[i];
					vector[i]+= (lvector[1] - vWorldPiv[1])*vWorldRSY[i];
					vector[i]+= (lvector[2] - vWorldPiv[2])*vWorldRSZ[i];
					vector[i]+= vWorldPos[i];
				}
//			}

			//*** Overall scale
			vector[0] = vector[0] * instdata->m_fScaleFactor;
			vector[1] = vector[1] * instdata->m_fScaleFactor;
			vector[2] = vector[2] * instdata->m_fScaleFactor;

			//*** Push the index of this vertex onto the vertex index
			vindex.push_back(verts.size());

			//*** Push this vertex onto the vertex stack.  The LWPoint class takes
			//*** care of axis conversion between LW and DTS
			verts.push_back (LWPoint(vector)) ;

			//*** TODO: The intial MilkShape code from which this derived from only
			//*** allows for a single bone per vertex (as MS only allows
			//*** that many).  LW supports many bones per vertex, and this
			//*** will need to be coded here.

			//*** Check if there is a weight mapped bone for this vertex.  If so, set that
			//*** bone as the node for this vertex.  Otherwise, set the root node as the
			//*** node for this vertex.  For the moment, only one bone may control a vertex,
			//*** so the weight is set to 1.0.
			char tempbuffer[1024];
			char* wmap = odb.GetPointFirstWeightMap(v);
			if(wmap == NULL)
			{
				//*** There is no weight map for this vertex, so set the root bone as its node
				vbone.push_back(getVertexBone(rootBone));
			} else
			{
				//*** Track down the first bone that uses this weight map
				int bonenode = instdata->FindMatchingBoneNode(itemID, wmap);
				if(bonenode >= 0)
				{
					//*** Found a bone node that matches
					vbone.push_back(getVertexBone(bonenode));
				} else
				{
					//*** No bone matches, so use the root node
					vbone.push_back(getVertexBone(rootBone));
				}
			}
			vweight.push_back(1.0);

			//*** Build texture coordinates
			double texUV[2];
			odb.GetPointUV(v, nPPType, texUV);
			texUV[1] = 1.0 - texUV[1];
			tverts.push_back (Point2D(texUV[0], texUV[1])) ;

		}

		//*** Adjust our type. We default to a rigid mesh, but if the mesh
		//*** is attached to more than one bone, we need to output as a skin
		//*** mesh. Node index is managed by the getVertexBone method.
		if (nodeIndex.size() > 1)
			setType(T_Skin);

		//*** Import primitives (as single triangles for now)
		//*** While we're at it, use the triangle data to get per-vertex normals
      
		std::vector<bool> vertexNormalFound (verts.size(), false) ;    
		normals.assign  (verts.size(), Point()) ;
		enormals.assign (verts.size(), '\0') ;

		for (v = 0 ; v < odb.m_nPolys ; v++)
		{
			char tempbuffer[1024];

			//*** Skip any polygon that has less than 3 points
			if(odb.m_pPoly[v].nverts < 3)
			{
				continue;
			}

			Primitive    myTriangle ;

			//*** Store the normals for each vertex in a polygon
			for (n = 0 ; n < 3 ; n++)
			{
				normals[odb.m_pPoly[v].v[n].index] = Point(odb.m_pPoly[v].v[n].norm[nPPType][0], odb.m_pPoly[v].v[n].norm[nPPType][2], odb.m_pPoly[v].v[n].norm[nPPType][1] );
				enormals[odb.m_pPoly[v].v[n].index] = encodeNormal(Point(odb.m_pPoly[v].v[n].norm[nPPType][0], odb.m_pPoly[v].v[n].norm[nPPType][2], odb.m_pPoly[v].v[n].norm[nPPType][1] ) );
			}

			//*** Create a triangle primitive for this triangle
			myTriangle.firstElement = indices.size() ;
			myTriangle.numElements  = 3 ;
			myTriangle.type         = Primitive::Strip | Primitive::Indexed ;

			int surfaceIndex = odb.m_pPoly[v].sindex;
			materialIndex = instdata->GetSurfaceIndex(odb.m_pSurfs[surfaceIndex].id);

			if ( (instdata->GetObjectExportType(itemID) != CLWData::ET_Collision) && (instdata->m_nExportMaterials == 1) && (materialIndex >= 0) )
				myTriangle.type |= materialIndex ;
			else
				myTriangle.type |= Primitive::NoMaterial ;

			indices.push_back  (odb.m_pPoly[v].v[0].index) ;
			indices.push_back  (odb.m_pPoly[v].v[1].index) ;
			indices.push_back  (odb.m_pPoly[v].v[2].index) ;

			primitives.push_back (myTriangle) ;
		}

		// Other stuff we don't support
      
		setFrames(1) ;
		setParent(-1) ;

		calculateBounds() ;
		calculateCenter() ;
		calculateRadius() ;

	}
}