/*
**	Revision History:
**	February 11, 2002	David Wyand		Began changes to convert for LW use
**	March 28, 2002		David Wyand		Added the forcing of the 'no environmental reflections'
**										flag if a LW surface's reflection is set to zero.  Also
**										changed the default Torque transparency setting to no
**										longer be additive.
**	April 25, 2002		David Wyand		Added support for the special IK motion handler.  This
**										allows IK data to be output in the DTS animation format.
*/

#pragma warning ( disable: 4786 )
#include <stack>
#include <string>
#include <vector>
#include <map>
#include <set>
#include <cmath>
#include <cassert>

#include "LWGlobal.h"
#include "LWObjectDB.h"

#include "dtsSDK/DTSShape.h"
#include "dtsSDK/DTSBrushMesh.h"
#include "LWESData.h"
#include "DTSLWMesh.h"
#include "DTSLWShape.h"

#include "Debug.h"

//#include "msLib.h"

//*** Prototypes
void StripTextureName(char* textureName);

namespace DTS
{

   // -----------------------------------------------------------------------

	//*** TODO: This is the old MilkShape mesh flag function.  Remove for LW
   int extractMeshFlags(char * name)
   {
      // Extract comma seperated flags embeded in the name. Anything after
      // the ":" in the name is assumed to be flag arguments and stripped
      // off the name.
      int flags = 0;
      char *ptr = strstr(name,":");
      if (ptr)
      {
         *ptr = 0;
         for (char *tok = strtok(ptr+1,","); tok != 0; tok = strtok(0,","))
         {
            // Strip lead/trailing white space
            for (; isspace(*tok); tok++) ;
            for (char* itr = tok + strlen(tok)-1; itr > tok && isspace(*itr); itr--)
               *itr = 0;

            //
            if (!stricmp(tok,"billboard"))
               flags |= Mesh::Billboard;
               else
            if (!stricmp(tok,"billboardz"))
               flags |= Mesh::Billboard | Mesh::BillboardZ;
               else
            if (!stricmp(tok,"enormals"))
               flags |= Mesh::EncodedNormals;
         }
      }
      return flags;
   }

   // -----------------------------------------------------------------------
   //*** TODO: This is a MilkShape function for checking if a bone is animated.
   //*** Likely not needed for LW, but leave as a reference for now.
/*+++

   bool isBoneAnimated(msBone *bone, int start, int end)
   {
      // Returns true if the bone contains any key frames
      // within the given time range.
      int numPKeys  = msBone_GetPositionKeyCount(bone);
      for (int j = 0 ; j < numPKeys ; j++) {
         msPositionKey * msKey = msBone_GetPositionKeyAt (bone, j);
         // MS is one based, start/end 0 based..
         if (msKey->fTime > start && msKey->fTime <= end)
            return true;
      }
      int numRKeys  = msBone_GetRotationKeyCount(bone);
      for (int i = 0 ; i < numRKeys ; i++)
      {
         msRotationKey * msKey = msBone_GetRotationKeyAt (bone, i);
         // MS is one based, start/end 0 based..
         if (msKey->fTime > start && msKey->fTime <= end)
            return true;
      }
      return false;
   }
+++*/

   // -----------------------------------------------------------------------
   //  Imports a Lightwave Model
   // -----------------------------------------------------------------------

   CLWShape::ImportConfig::ImportConfig ()
   {
      withMaterials       = true; //+++ true ;
      collisionMesh       = -1 ;
      collisionType       = C_BBox; //+++ C_None ;
      collisionComplexity = 0.2f ; 
      collisionVisible    = false ;
      animation           = false; //+++ true ;
      reset();
   }

   void CLWShape::ImportConfig::reset()
   {
      // Reset values that must be specified using opt: or seq:
      scaleFactor         = 1.0f; //+++ 0.1f ;
      minimumSize         = 0 ;
      animationFPS        = 15 ;
      animationCyclic     = false;
      sequence.resize(0);
   }


	CLWShape::CLWShape (CLWData* instdata) 
	{
		int		i;
		int		reservedMesh = -1 ;
		int		nRootNode = -1;
		char	buffer[512] ;

		// --------------------------------------------------------
		//  Materials (optional)
		// --------------------------------------------------------

		if (instdata->m_nExportMaterials == 1)
		{
			CLWGlobal::LayoutMonitorStep(1,"Building materials");

			//*** Here's what to do:
			//*** Go through all of the objects in the scene and build up a global
			//*** surface list.  Store this list.  Then, when an object refers to
			//*** a surface, it needs to be related back to this global list for
			//*** Torque.  Then, during Mesh creation, each Triangle will point
			//*** to this global list rather than the Mesh surface list (see the
			//*** triangle building in the DTSLWMesh.cpp file on where this
			//*** look-up needs to occur).

			//*** Add LW surfaces to a global list for look-up, and build
			//*** the DTS material.
			char textureName[1024];
			LWSurfaceID surf;
			surf = CLWGlobal::surff->first();
			while(surf)
			{
				LWTextureID tex;
				LWTLayerID tlayer;
				int type;
				LWImageID* image;
				double *dval;

				//*** Default texture name
				strcpy(textureName, "Unnamed");

				//*** Obtain the first colour texture
				tex = CLWGlobal::surff->getTex( surf, SURF_COLR );
				if(tex)
				{
					tlayer = CLWGlobal::txtrf->firstLayer( tex );
					type = CLWGlobal::txtrf->layerType( tlayer );
					if ( type == TLT_IMAGE )
					{
						CLWGlobal::txtrf->getParam( tlayer, TXTAG_IMAGE, &image );
						if(image)
						{
							//*** Get the filename of the texture
							strncpy(textureName, CLWGlobal::imglist->filename(image,0),1024);

							//*** Strip texture file extension
							StripTextureName(textureName);
/*							char * ptr = textureName + strlen(textureName) ;
							while (ptr > textureName && *ptr != '.' && *ptr != '\\' && *ptr != '/' && *ptr != ':')
							{
								ptr-- ;
							}
							if (*ptr == '.')
							{
								*ptr = '\0' ;
							}

							//*** Strip texture file path 
							ptr = textureName + strlen(textureName) ;
							while (ptr > textureName && *ptr != '\\' && *ptr != '/' && *ptr != ':')
							{
								ptr-- ;
							}
							if (*ptr == '\\' || *ptr == '/' || *ptr == ':')
							{
								memmove (textureName, ptr+1, strlen(ptr)+1) ;
							}
*/
						}
					}
				}

				//*** Create the material
				Material mat ;
				mat.name        = textureName ;
//				mat.flags       = Material::SWrap | Material::TWrap | Material::NeverEnvMap ;
				mat.flags       = Material::SWrap | Material::TWrap;
				mat.reflectance = materials.size() ;	// This will point to this material, or to the reflection material if it is defined below
				mat.bump        = -1 ;
				mat.detail      = -1 ;
				mat.reflection  = 0.0f ;
				mat.detailScale = 1.0f ;

				//*** Add flags based on Material Attributes shader
				CLWESData* materialattr = CLWESData::FindBySurface(surf);
				if(materialattr != NULL)
				{
					materialattr->UpdateMaterialFlags(mat.flags);
				}
				
				//*** Handle luminosity value
				dval = CLWGlobal::surff->getFlt( surf, SURF_LUMI );
				if( *dval > 0.0)
				{
					mat.flags |= Material::SelfIlluminating ;
				}
				
				//*** Check transparency value
				//+++ if (msMaterial_GetTransparency (msMat) <= 0.75f)
				dval = CLWGlobal::surff->getFlt( surf, SURF_TRAN );
				if( *dval > 0.0)
				{
//					mat.flags |= Material::Translucent | Material::Additive ;
					mat.flags |= Material::Translucent ;
				}

				//*** Handle reflection value
				dval = CLWGlobal::surff->getFlt( surf, SURF_REFL );
				if( *dval > 0.0)
				{
					mat.reflection  = (*dval);
				} else
				{
					//*** Disable environmental mapping
					mat.flags |= Material::NeverEnvMap;
				}

				//*** Check if a separate reflection image is being used and
				//*** set up the DTS file to use it
				//*** TODO: Does not appear to be working all that well.  Check.
				tex = CLWGlobal::surff->getTex( surf, SURF_REFL );
				if(tex)
				{
					tlayer = CLWGlobal::txtrf->firstLayer( tex );
					type = CLWGlobal::txtrf->layerType( tlayer );
					if ( type == TLT_IMAGE )
					{
						CLWGlobal::txtrf->getParam( tlayer, TXTAG_IMAGE, &image );
						if(image)
						{

							//*** There is a reflection image so make use of it
							char reftextureName[1024];

							//*** Get the filename of the texture
							strncpy(reftextureName, CLWGlobal::imglist->filename(image,0),1024);

							//*** Strip texture file extension
							StripTextureName(reftextureName);

							//*** Build the material for the reflection
							Material refmat ;
							refmat.name        = reftextureName ;
							refmat.flags       = Material::SWrap | Material::TWrap;
							refmat.reflectance = materials.size() ;
							refmat.bump        = -1 ;
							refmat.detail      = -1 ;
							refmat.reflection  = (*dval);
							refmat.detailScale = 1.0f ;

							materials.push_back(refmat) ;

							//*** Add this new material to the surface list, but without
							//*** a LightWave surface ID as it does not truely reflect
							//*** a LW surface
							instdata->AddSurface(NULL);

						}
					}
				}

				//*** Add the surface ID to the global (scene-level) list
				instdata->AddSurface(surf);

				//*** Push this material onto the material stack
				materials.push_back(mat) ;

				surf = CLWGlobal::surff->next(surf);
			}

		} else
		{
			CLWGlobal::LayoutMonitorStep(1,"Materials not exported");
		}

		// --------------------------------------------------------
		//  Nodes & Bones
		// --------------------------------------------------------

		CLWGlobal::LayoutMonitorStep(1,"Building nodes");
		Node n ;
		assert(!nodes.size());

		//*** Build up the node list
		instdata->BuildNodeList();

		//*** Now loop through the LW specific node list and build
		//*** the DTS node list
		for(int nodenum = 0; nodenum < instdata->GetNumberOfNodes(); ++nodenum)
		{
			n.parent = instdata->GetNodeParent(nodenum);
			n.name = addName(instdata->GetNodeName(nodenum));
			nodes.push_back (n) ;

			nodeDefTranslations.push_back(LWPoint(instdata->GetNodePosition(nodenum)) * instdata->m_fScaleFactor) ;
			nodeDefRotations.push_back(LWQuaternion(instdata->GetNodeRotation(nodenum))) ;
		}


		//*** Handle the Root Node if it exists (-1 if it doesn't)
		nRootNode = instdata->m_nRootNode;


		// --------------------------------------------------------
		//  Mesh objects.
		// --------------------------------------------------------

		Object o ;
		
		//*** Obtain the number of meshes
		int numMeshes = instdata->GetObjectCount(); // CLWGlobal::GetObjectCount();

		for (i = 0 ; i < numMeshes ; i++)
		{

			//*** Obtain the LW item ID based of the absolute object number (i)
			LWItemID objID = instdata->m_vObjectList[i]->m_ID; // CLWGlobal::GetObjectByNum(i);

			sprintf(buffer,"Working on mesh %i (%s)",i,instdata->m_vObjectList[i]->m_szLWName);
			CLWGlobal::LayoutMonitorStep(1,buffer);

			//*** Only work on a mesh that is set to export and has polygons
			if( (instdata->GetObjectExportType(objID) != CLWData::ET_Mesh) || (CLWGlobal::objinfo->numPolygons(objID) <= 0) )
			{
				continue;
			}

			//*** Obtain the name of object and place it into the buffer.  This will
			//*** use the LW name if the user has not provided a specific DTS name
			if(instdata->m_vObjectList[i]->m_szDTSName[0] == '\0')
			{
				strncpy(buffer,instdata->m_vObjectList[i]->m_szLWName,MAX_OBJECT_NAME);
			} else
			{
				strncpy(buffer,instdata->m_vObjectList[i]->m_szDTSName,MAX_OBJECT_NAME);
			}


			//*** Get object struct ready.  Objects are entities that
			//*** represent renderable items. Objects can have more
			//*** than one mesh to represent different detail levels.
			o.name      = addName(buffer) ;
			o.numMeshes = 1 ;
			o.firstMesh = meshes.size() ;
			if(instdata->GetNodeNumber(objID) >= 0)
			{
				//*** Use the mesh's node for this mesh
				o.node = instdata->GetNodeNumber(objID);
			} else
			{
				//*** Use the root node for this mesh
				assert(nRootNode != -1);
				o.node = nRootNode; //nodes.size() - 1;
			}

			// Process the raw data.
			//*** NOTE: 'o.node' below points to the 'Root' bone last defined above in the bone section!
			//*** This is because 'nodes.size()-1' is equal to the zero indexed node added to the 'nodes' vector.
			//*** In this case, because the 'Root' node is added last, it will be used as the default bone.
			meshes.push_back(CLWMesh (instdata, objID, o.node)) ;
			Mesh& m = meshes.back();

			//*** Set the mesh flags
			int flags = 0;
			if(instdata->m_vObjectList[i]->m_nGlobalRotation == 1)
			{
				flags |= Mesh::Billboard;
			}
			if(instdata->m_vObjectList[i]->m_nGlobalRotation == 2)
			{
				flags |= Mesh::Billboard | Mesh::BillboardZ;
			}
			if(instdata->m_vObjectList[i]->m_nEncodedNormals == 1)
			{
				flags |= Mesh::EncodedNormals;
			}
			m.setFlag(flags);

			//*** Rigid meshes can be attached to a single node, in which
			//*** case we need to transform the vertices into the node's
			//*** local space.
			if (m.getType() == Mesh::T_Standard)
			{
CLWGlobal::LayoutMonitorStep(0,"-- This is a Standard Mesh");

				o.node = m.getNodeIndex(0);
				if (o.node != -1)
				{
					//*** Transform the mesh into node space.  The mesh vertices
					//*** must all be relative to the bone their attached to.
					Quaternion world_rot;
					Point world_trans;
					getNodeWorldPosRot(o.node,world_trans, world_rot);
					m.translate(-world_trans);
					m.rotate(world_rot.inverse());
				}
			}

			//*** Skin meshes need transform information to be able to
			//*** transform vertices into bone space (should fix the Torque
			//*** engine to calculate these at load time).
			if (m.getType() == Mesh::T_Skin)
			{
CLWGlobal::LayoutMonitorStep(0,"-- This is a Skin Mesh");
				for (int n = 0; n < m.getNodeIndexCount(); n++)
				{
					Quaternion world_rot;
					Point world_trans;
					getNodeWorldPosRot(m.getNodeIndex(n),world_trans, world_rot);
					m.setNodeTransform(n,world_trans,world_rot);
				}
			}

			objects.push_back(o) ;
		}

		//*** --------------------------------------------------------
		//***  Detail levels 
		//***  Not fully supported, we'll just export all our current
		//***  stuff as single detail level.
		//*** --------------------------------------------------------

		DetailLevel dl ;
		dl.name         = addName("Detail-1") ;
		dl.size         = 0 ; // Projected size of detail level
		dl.objectDetail = 0 ;
		dl.polyCount    = 0 ;
		dl.subshape     = 0 ;
		dl.maxError     = -1 ;
		dl.avgError     = -1 ;

		std::vector<Mesh>::iterator mesh_ptr ;
		for (mesh_ptr = meshes.begin() ; mesh_ptr != meshes.end() ; mesh_ptr++)
		{
			dl.polyCount += mesh_ptr->getPolyCount() ;
		}

		detailLevels.push_back(dl) ;

		//*** --------------------------------------------------------
		//***  Collision detail levels and meshes (optional)
		//*** --------------------------------------------------------

		//*** Update our bounds values as we might use them to build
		//*** a collision mesh.
		calculateBounds() ;
		calculateCenter() ;
		calculateRadius() ;
		calculateTubeRadius() ;

		if (instdata->m_nCollisionType != 0)
		{
			CLWGlobal::LayoutMonitorStep(1,"Building collision objects");

			//*** Create a temporary collision mesh
			Mesh * collmesh ;
			LWItemID objID;
			int nColNode = instdata->m_nRootNode;	//*** Set the collision's node to the Root node by default

			Box myBoundingBox(getBounds());
			switch (instdata->m_nCollisionType)
			{
				case CLWData::CT_BBox:
					//*** Bounding Box
					assert(nColNode != -1);
					collmesh = new BoxMesh(myBoundingBox) ;
					break ;
				case CLWData::CT_BCylinder:
					//*** Bounding Cylinder
					assert(nColNode != -1);
					collmesh = new CylinderMesh(myBoundingBox, getTubeRadius(), instdata->m_fCollisionComplexity) ;
					break ;
				case CLWData::CT_Mesh:
					//*** Meshes provide collision
					objID = instdata->GetFirstCollisionObject();
					if(instdata->GetNodeNumber(objID) >= 0)
					{
						//*** Use the mesh's node for this mesh
						nColNode = instdata->GetNodeNumber(objID);
					} else
					{
						assert(nColNode != -1);
					}
					collmesh = new CLWMesh (instdata, objID, nColNode);
					break ;
				default:
					assert (0 && "Invalid collision mesh type") ;
			}
         
			//*** Create the collision detail level
			dl.name         = addName("Collision-1") ;
			dl.size         = -1 ; // -1 sized detail levels are never rendered
			dl.objectDetail = detailLevels.size() ;
			dl.subshape     = 0 ;
			dl.polyCount    = collmesh->getPolyCount() ;
			detailLevels.push_back(dl) ;
         
			//*** Create an object for the collision mesh and attach it to
			//*** the "Root" node.
			o.name         = addName("Col") ;
			o.node         = nColNode; //nodes.size() - 1 ;
			o.firstMesh    = meshes.size() ;
			o.numMeshes    = dl.objectDetail + 1 ;
			objects.push_back(o) ;

			if (instdata->m_nCollisionVisible == 1)
			{
				//*** Create a renderable copy of the collision meshes
				//*** for visible detail levels (with traslucent material)

				for (i = 0 ; i < dl.objectDetail ; i++)
				{
					meshes.push_back(*collmesh) ;
					meshes[meshes.size()-1].setMaterial(materials.size()) ;
				}

				Material mat ;
				mat.name        = "Collision_Mesh" ;
				mat.flags       = Material::NeverEnvMap | Material::SelfIlluminating;
				mat.reflectance = materials.size() ;
				mat.bump        = -1 ;
				mat.detail      = -1 ;
				mat.reflection  = 1.0f ;
				mat.detailScale = 1.0f ;
				materials.push_back(mat) ;
			} else
			{
				//*** Create null meshes so the collision object
				//*** is not rendered in visible detail levels

				for (i = 0 ; i < dl.objectDetail ; i++)
				{
					meshes.push_back(Mesh(Mesh::T_Null)) ;
				}
			}

			//*** Copy the collision mesh and destroy the temporary
			meshes.push_back(*collmesh) ;
			switch (instdata->m_nCollisionType)
			{
				case CLWData::CT_BBox:		delete (BoxMesh *)collmesh ; break ;
				case CLWData::CT_BCylinder:	delete (CylinderMesh *)collmesh ; break ;
				//+++ case C_Mesh:     delete (MilkshapeMesh *)collmesh ; break ;
				case CLWData::CT_Mesh:		delete (CLWMesh *)collmesh ; break ;
			}
		} else
		{
			CLWGlobal::LayoutMonitorStep(1,"Collision objects not built");
		}

		//*** --------------------------------------------------------
		//***  Subshape and final stuff
		//*** --------------------------------------------------------

		//*** Create a subshape with everything

		Subshape s ;
		s.firstObject = 0 ;
		s.firstNode   = 0 ;
		s.firstDecal  = 0 ;
		s.numNodes    = nodes.size() ;
		s.numObjects  = objects.size() ;
		s.numDecals   = decals.size() ;
		subshapes.push_back(s) ;

		//*** Create an object state for each object (not sure about this)

		ObjectState os ;
		os.vis      = 1.0f ;
		os.frame    = 0 ;
		os.matFrame = 0 ;
		for (i = 0 ; i < objects.size() ; i++)
		{
			objectStates.push_back(os) ;
		}

		//*** Recalculate bounds (they may have changed)

		calculateBounds() ;
		calculateCenter() ;
		calculateRadius() ;
		calculateTubeRadius() ;

		setSmallestSize(instdata->m_nMinPixelSize) ;

      // --------------------------------------------------------
      //  Animation (optional)

      // For each sequence, Torque wants an array of frame * nodes
      // information for all nodes affected by that sequence. Node
      // animation information can be translation, rotation, scale,
      // visibility, material, etc.
      //
      // The sequence contains a "matters" array for each type of
      // animation info.  Each type has it's own array of frame * nodes
      // which contains only the nodes affected by that type of
      // information for the sequence.  Since each array is NxN,
      // if a node is animated on a single frame of the sequence, it
      // will get an entry for every frame.

      // --------------------------------------------------------

		if(instdata->m_nExportAnimSequences == 1)
		{
			CLWGlobal::LayoutMonitorStep(1,"Building animation sequences");

			//*** Run through each of the frames to build IK channels for special IK plug-in
			CLWGlobal::LayoutMonitorStep(0,"Building each frame for IK");
			LWFrame fstart = CLWGlobal::ui->previewStart;
			LWFrame fend = CLWGlobal::ui->previewEnd;
			for(LWFrame curframe=fstart; curframe <= fend; ++curframe)
			{
				char buffer[256];
				sprintf(buffer,"GoToFrame %i",curframe);
				CLWGlobal::evaluate(buffer);
			}

			int nCount = instdata->GetAnimSequenceCount();
			for(int sc = 0; sc < nCount; ++sc)
			{
				int nodenum = 0;
				int nodeCount = 0;
				int framenum = 0;

				//*** Build the sequence structure for exporting
				Sequence s;
				s.flags				= Sequence::UniformScale;
				s.nameIndex			= addName(instdata->m_vSequences[sc]->m_szName) ;
				s.numKeyFrames		= (instdata->m_vSequences[sc]->m_nEnd) - (instdata->m_vSequences[sc]->m_nStart);
				s.duration			= float(s.numKeyFrames) / (instdata->m_vSequences[sc]->m_nFPS);
				s.baseTranslation	= nodeTranslations.size() ;
				s.baseRotation		= nodeRotations.size();

				//*** Setup additional flags for the sequence
				if(instdata->m_vSequences[sc]->m_nCyclic == 1)
				{
					s.flags |= Sequence::Cyclic;
				}

				//*** Count how many nodes are affected by the sequence and
				//*** set the sequence.matter arrays to indicate which ones.
				s.matters.translation.assign (nodes.size(), false);
				s.matters.rotation.assign    (nodes.size(), false);
				for (nodenum = 0 ; nodenum < instdata->GetNumberOfNodes() ; ++nodenum)
				{
					//*** For the moment, mark all nodes as having positional
					//*** and rotational keyframes for all sequences.  This will
					//*** be changed so that only those nodes that move or rotate
					//*** during this sequence will be marked as such.
					//*** TODO: Allow the user to choose which nodes should export
					//*** rotation and translation information to the DTS file.
					s.matters.translation[nodenum] = true;
					s.matters.rotation[nodenum] = true;
					nodeCount++;
				}

				//*** Size arrays to hold keyframe * nodeCount
				nodeTranslations.resize (nodeTranslations.size() + nodeCount * s.numKeyFrames);
				nodeRotations.resize (nodeTranslations.size());

				//*** Set the keyframe data for each affected bone.  Unaffected
				//*** bones are skipped so the final NxN array of transforms
				//*** and rotations is "compressed", sort of.
				int index = 0;
				for (nodenum = 0 ; nodenum < instdata->GetNumberOfNodes() ; ++nodenum)
				{
					//*** For now, all nodes are animated for all keyframes and sequences

					//*** Insert translation keys into the table.
					Point *translations = &nodeTranslations[s.baseTranslation + index * s.numKeyFrames];
					for(framenum = 0; framenum < s.numKeyFrames; ++framenum)
					{
						//*** Write out the position of the node for each frame in the sequence
						LWDVector lwvec;
						float fvec[3];

						double time = ((double)(instdata->m_vSequences[sc]->m_nStart + framenum)) / CLWGlobal::scninfo->framesPerSecond;

						CLWGlobal::iteminfo->param((instdata->m_vNodes[nodenum]->m_ID), LWIP_POSITION, time, lwvec);
						fvec[0] = (float)lwvec[0];
						fvec[1] = (float)lwvec[1];
						fvec[2] = (float)lwvec[2];

//						translations[framenum] = nodeDefTranslations[nodenum] +
//							nodeDefRotations[nodenum].apply(LWPoint(fvec) * instdata->m_fScaleFactor);
//						translations[framenum] = nodeDefRotations[nodenum].apply(LWPoint(fvec) * instdata->m_fScaleFactor);
						translations[framenum] = LWPoint(fvec) * instdata->m_fScaleFactor;

					}

					//*** Insert rotation keys into the table.
					Quaternion *rotations = &nodeRotations[s.baseTranslation + index * s.numKeyFrames];
					for(framenum = 0; framenum < s.numKeyFrames; ++framenum)
					{
						//*** Write out the rotation of the node for each frame in the sequence
						LWDVector lwvec;
						float fvec[3];
						LWChanGroupID	chanGroup;
						int				motType[3];
						LWEnvelopeID	envid;

						double time = ((double)(instdata->m_vSequences[sc]->m_nStart + framenum)) / CLWGlobal::scninfo->framesPerSecond;

						//*** Get keyframed rotations
						CLWGlobal::iteminfo->param((instdata->m_vNodes[nodenum]->m_ID), LWIP_ROTATION, time, lwvec);

						//*** Get the motion type and channel group for the node
						CLWGlobal::iteminfo->controller((instdata->m_vNodes[nodenum]->m_ID), LWIP_ROTATION, motType);
						chanGroup = CLWGlobal::iteminfo->chanGroup(instdata->m_vNodes[nodenum]->m_ID);

						//*** If a DTSIK.H channel exists, use it's value, otherwise use the keyframe
						if(motType[0] == LWMOTCTL_IK)
						{
							envid = CLWGlobal::findEnv( chanGroup, "DTSIK.H" );
							if(envid)
							{
								//*** Use data from special channel
								double value = CLWGlobal::envfunc->evaluate( envid, time );
								fvec[0] = (float)value;
							//	LWEnvKeyframeID keyid = CLWGlobal::envfunc->createKey( envid, ima->time, rot[0] );
							} else
							{
								//*** Use keyframe
								fvec[0] = (float)lwvec[0];
							}
						} else
						{
							//*** Use keyframe
							fvec[0] = (float)lwvec[0];
						}

						//*** If a DTSIK.P channel exists, use it's value, otherwise use the keyframe
						if(motType[1] == LWMOTCTL_IK)
						{
							envid = CLWGlobal::findEnv( chanGroup, "DTSIK.P" );
							if(envid)
							{
								//*** Use data from special channel
								double value = CLWGlobal::envfunc->evaluate( envid, time );
								fvec[1] = (float)value;
							//	LWEnvKeyframeID keyid = CLWGlobal::envfunc->createKey( envid, ima->time, rot[0] );
							} else
							{
								//*** Use keyframe
								fvec[1] = (float)lwvec[1];
							}
						} else
						{
							//*** Use keyframe
							fvec[1] = (float)lwvec[1];
						}

						//*** If a DTSIK.B channel exists, use it's value, otherwise use the keyframe
						if(motType[2] == LWMOTCTL_IK)
						{
							envid = CLWGlobal::findEnv( chanGroup, "DTSIK.B" );
							if(envid)
							{
								//*** Use data from special channel
								double value = CLWGlobal::envfunc->evaluate( envid, time );
								fvec[2] = (float)value;
							//	LWEnvKeyframeID keyid = CLWGlobal::envfunc->createKey( envid, ima->time, rot[0] );
							} else
							{
								//*** Use keyframe
								fvec[2] = (float)lwvec[2];
							}
						} else
						{
							//*** Use keyframe
							fvec[2] = (float)lwvec[2];
						}

//						rotations[framenum] = LWQuaternion(fvec) * nodeDefRotations[nodenum];
						rotations[framenum] = LWQuaternion(fvec);
					}

					//*** Increment the index for the pointers into the translation & rotation
					//*** arrays
					index++;
				}

				//*** Push this sequence onto the array
				sequences.push_back(s);
			}

		} else
		{
			CLWGlobal::LayoutMonitorStep(1,"Animation Sequences not exported");
		}


/*+++ MilkShape method for sequences, kept for reference for now
      int frameCount = msModel_GetTotalFrames (model);
      if (frameCount && numBones && config.animation)
      {
         // Process all the sequences.
         for (int sc = 0; sc < config.sequence.size(); sc++)
         {
            MilkshapeShape::ImportConfig::Sequence& si = config.sequence[sc];

            // Build the exported sequence structure
            Sequence s;
            s.flags           = Sequence::UniformScale; 
            if (si.cyclic)
               s.flags |= Sequence::Cyclic;
            s.nameIndex       = addName(si.name) ;
            s.numKeyFrames    = si.end - si.start;
            s.duration        = float(s.numKeyFrames) / si.fps;
            s.baseTranslation = nodeTranslations.size() ;
            s.baseRotation    = nodeRotations.size();

            // Count how many nodes are affected by the sequence and
            // set the sequence.matter arrays to indicate which ones.
            s.matters.translation.assign (nodes.size(), false);
            s.matters.rotation.assign    (nodes.size(), false);
            int nodeCount = 0;
            for (i = 0 ; i < numBones ; i++)
            {
               msBone * bone = msModel_GetBoneAt(model, i);
               if (isBoneAnimated(bone,si.start,si.end))
               {
                  // Milkshape seems to always produce rotation & position
                  // keys in pairs, so we'll just deal with them together.
                  s.matters.translation[i] = true;
                  s.matters.rotation[i] = true;
                  nodeCount++;
               }
            }

            // Size arrays to hold keyframe * nodeCount
            nodeTranslations.resize (nodeTranslations.size() + nodeCount * s.numKeyFrames);
            nodeRotations.resize (nodeTranslations.size());

            // Set the keyframe data for each affected bone.  Unaffected
            // bones are skipped so the final NxN array of transforms
            // and rotations is "compressed", sort of.  We could compress
            // both the rotation and the position arrays individually,
            // but MS seems to always generate the values in pairs, so
            // we'll do them together. Though the msKey loops are seperate,
            // just in case.
            int index = 0;
            for (i = 0 ; i < numBones ; i++)
            {
               msBone * bone = msModel_GetBoneAt(model, i) ;
               if (isBoneAnimated(bone,si.start,si.end))
               {
                  int numPKeys  = msBone_GetPositionKeyCount(bone);
                  int numRKeys  = msBone_GetRotationKeyCount(bone);

                  // Insert translation keys into the table.
                  Point *translations = &nodeTranslations[s.baseTranslation + index * s.numKeyFrames];
                  int lastFrame = 0;

                  for (j = 0 ; j < numPKeys ; j++)
                  {
                     msPositionKey * msKey = msBone_GetPositionKeyAt (bone, j);
                     int frame = int(msKey->fTime) - 1;

                     // Only want keys in the sequence range. If it's before
                     // our range, we'll put it into the 0 frame in case we don't
                     // get a key for that frame later.
                     if (frame >= si.end)
                        break;
                     if (frame < si.start)
                        frame = 0;
                     else
                        frame -= si.start;

                     // Store the total translation for the node and fill in
                     // the initial frame if this is the first key frame.
                     translations[frame] = nodeDefTranslations[i] +
                        nodeDefRotations[i].apply(MilkshapePoint(msKey->Position) * config.scaleFactor);
                     if (!j && frame > 0)
                        translations[0] = translations[frame];

                     // Using the last translation to fill in missing frames.
                     for (int f = lastFrame + 1; f < frame; f++)
                        translations[f] = translations[lastFrame];
                     lastFrame = frame;
                  }

                  // Duplicate the last frame to the end.
                  for (int t = lastFrame + 1; t < s.numKeyFrames; t++)
                     translations[t] = translations[lastFrame];
   
                  // Insert rotation keys into the table.
                  Quaternion *rotations = &nodeRotations[s.baseTranslation + index * s.numKeyFrames];
                  lastFrame = 0;

                  for (j = 0 ; j < numRKeys ; j++)
                  {
                     msRotationKey * msKey = msBone_GetRotationKeyAt (bone, j) ;
                     int frame = int(msKey->fTime) - 1;
   
                     // Only want keys in the sequence range. If it's before
                     // our range, we'll put it into the 0 frame in case we don't
                     // get a key for that frame later.
                     if (frame >= si.end)
                        break;
                     if (frame < si.start)
                        frame = 0;
                     else
                        frame -= si.start;

                     // Store the total rotation for the node and fill in
                     // the initial frame if this is the first key frame.
                     rotations[frame] = MilkshapeQuaternion(msKey->Rotation) * nodeDefRotations[i];
                     if (!j && frame > 0)
                        rotations[0] = rotations[frame];

                     // Fill in any missing frames
                     for (int f = lastFrame + 1; f < frame; f++)
                        rotations[f] = rotations[lastFrame];
                     lastFrame = frame;
                  }

                  // Duplicate the last frame to the end.
                  for (int r = lastFrame + 1; r < s.numKeyFrames; r++)
                     rotations[r] = rotations[lastFrame];

                  // Increment the position & rotation array index
                  index++;
               }
            }

            sequences.push_back(s);
         }
      }
+++*/

		//*** Clear out the global lists
		instdata->ClearNodeList();
		instdata->ClearSurfaces();
   }

} // DTS namespace

//*** Function to strip the extension and file path from a texture's filename
void StripTextureName(char* textureName)
{
	//*** Strip texture file extension
	char * ptr = textureName + strlen(textureName) ;
	while (ptr > textureName && *ptr != '.' && *ptr != '\\' && *ptr != '/' && *ptr != ':')
	{
		ptr-- ;
	}
	if (*ptr == '.')
	{
		*ptr = '\0' ;
	}

	//*** Strip texture file path 
	ptr = textureName + strlen(textureName) ;
	while (ptr > textureName && *ptr != '\\' && *ptr != '/' && *ptr != ':')
	{
		ptr-- ;
	}
	if (*ptr == '\\' || *ptr == '/' || *ptr == ':')
	{
		memmove (textureName, ptr+1, strlen(ptr)+1) ;
	}
}
