/*	GUIMESHSELECTOR.CPP
**
**	Supports the CGUIMeshSelector functions
**
**	Revision History:
**	February 11, 2002	David Wyand		Created file
**	March 12, 2002		David Wyand		Added the m_BaseTransFrameEdit control to allow the user
**										to choose which frame the node's position and rotation
**										will come from.
**	March 12, 2002		David Wyand		Added the 'View Nodes' button to the GUI to allow the
**										user to preview the node setup of the DTS object
**										prior to export.
**	April 4, 2002		David Wyand		Built out the Animation Tab.
**	April 25, 2002		David Wyand		Added the 'Setup for IK' button to the Animation Tab.
**										This button will add or remove the special IK Motion
**										handler from objects and bones if the item uses IK
**										to animate any of its rotation channels.
*/

//*** Includes

#pragma warning ( disable: 4786 )

#include <fstream> 

#include <lwmotion.h>	// To get the plug-in type name for the IK Motion Handler plug-in

#include "LWGlobal.h"
#include "DTSLWShape.h"
#include "GUIViewNodes.h"
#include "GUIMeshSelector.h"

#include "Debug.h"

//*** Prototypes
static void ExportButtonEvent( LWControl *ectl, void *edata );
static void CloseButtonEvent( LWControl *ectl, void *edata );
static void ViewNodesButtonEvent( LWControl *ectl, void *edata );

static int MeshListBoxCount( void *userdata );
static char* MeshListBoxName( void *userdata, int index, int column );
static int MeshListBoxColumnWidth( void *userdata, int column );
static void MeshListBoxEvent( LWControl *ectl, void *edata );
static void DTSNameEditEvent( LWControl *ectl, void *edata );
static void ExportTypePopupEvent( LWControl *ectl, void *edata );
static void GlobalRotationPopupEvent( LWControl *ectl, void *edata );
static void EncodedNormalsPopupEvent( LWControl *ectl, void *edata );

static void AnimExportChoiceEvent( LWControl *ectl, void *edata );
static void AnimIKSetupButtonEvent( LWControl *ectl, void *edata );
static int AnimListBoxCount( void *userdata );
static char* AnimListBoxName( void *userdata, int index, int column );
static int AnimListBoxColumnWidth( void *userdata, int column );
static void AnimationListBoxEvent( LWControl *ectl, void *edata );
static void AnimNewButtonEvent( LWControl *ectl, void *edata );
static void AnimDeleteButtonEvent( LWControl *ectl, void *edata );
static void AnimUpButtonEvent( LWControl *ectl, void *edata );
static void AnimDownButtonEvent( LWControl *ectl, void *edata );
static void AnimNameEditEvent( LWControl *ectl, void *edata );
static void AnimFPSEditEvent( LWControl *ectl, void *edata );
static void AnimStartEditEvent( LWControl *ectl, void *edata );
static void AnimEndEditEvent( LWControl *ectl, void *edata );
static void AnimCyclicChoiceEvent( LWControl *ectl, void *edata );

static void GlobalTabChoiceEvent( LWControl *ectl, void *edata );
static void SaveAsEvent( LWControl *ectl, void *edata );
static void ExportMaterialsChoiceEvent( LWControl *ectl, void *edata );
static void ScaleFactorEditEvent( LWControl *ectl, void *edata );
static void MinimunVisibleSizeEditEvent( LWControl *ectl, void *edata );
static void CollisionTypeChoiceEvent( LWControl *ectl, void *edata );
static void CollisionVisibleChoiceEvent( LWControl *ectl, void *edata );
static void CollisionComplexityEditEvent( LWControl *ectl, void *edata );

static void PolyPointTypeChoiceEvent( LWControl *ectl, void *edata );
static void BaseTransFrameEditEvent( LWControl *ectl, void *edata );
static void ExportMeshAsNodeChoiceEvent( LWControl *ectl, void *edata );
static void MeshNodeExtEditEvent( LWControl *ectl, void *edata );

static void PanelDrawCallback( LWPanelID panel, void *edata, DrMode mode );
static void PanelUserOpenCallback( LWPanelID panel, void *edata );
static void PanelUserCloseCallback( LWPanelID panel, void *edata );

//*** Global variables
const int MeshMaxColumns = 5;
char* MeshLBColumnHeadings[] =
{
	"Ex","LW Name","DTS Name","Rotation","Normals",NULL
};
int MeshLBColumnWidths[] =
{
	20,100,100,50,50
};
char* MeshLBExportType[] =
{
	"-","M","C","N",NULL
};
char* MeshLBRotation[] =
{
	"Normal","Billboard","BillboardZ",NULL
};
char* MeshLBEncodedNormals[] =
{
	"Standard","Encoded",NULL
};
const char* ExportTypePopupType[] =
{
	"Do not export","Mesh or Node","Collision Mesh","Node Only",NULL
};
const char* GlobalTabChoices[] =
{
	"Global","Animation","Advanced",NULL
};
const char * CollisionMeshChoices[] =
{
	"None","Bounding Box","Bounding Cylinder","Use Mesh",NULL
};
const char* GlobalRotationPopupType[] =
{
	"Normal","Billboard","BillboardZ",NULL
};
const char* EncodedNormalsPopupType[] =
{
	"Standard","Encoded",NULL
};
const char * PolyPointTypeChoices[] =
{
	"Normal","Transformed",NULL
};
const int AnimMaxColumns = 5;
char* AnimLBColumnHeadings[] =
{
	"Name","Start","End","FPS","Cyclic",NULL
};
int AnimLBColumnWidths[] =
{
	100,50,50,50,50
};
char* AnimLBCyclic[] =
{
	"No","Yes",NULL
};

//*** Constructor
CGUIMeshSelector::CGUIMeshSelector()
{

	m_nSelectedRow = -1;
	m_nAnimSelectedRow = -1;
	m_InstData = NULL;
}

//*** Destructor
CGUIMeshSelector::~CGUIMeshSelector()
{
//	vector<MeshSelectionNode*>::iterator itor;
//
//	//*** Delete the MeshSelectionNode instances.  Only need to do this for the m_ItemList
//	//*** as m_Nodes points to the same instances.
//	for(itor = m_ItemList.begin(); itor != m_ItemList.end(); ++itor)
//	{
//		delete (*itor);
//	}
//
//	m_ItemList.clear();

}

//*** Create the Mesh Selector
bool CGUIMeshSelector::CreateMeshSelector(CLWData* inst, LWPanelFuncs *panf, LWPanelID panel, int x, int y, int width, int height)
{
	LWPanControlDesc desc;
	static LWValue ival={LWT_INTEGER},ivecval={LWT_VINT},fval={LWT_FLOAT},fvecval={LWT_VFLOAT},sval={LWT_STRING};

	int pos;

	if(inst)
	{
		m_InstData = inst;
	} else
	{
		return false;
	}

	m_Panel = panel;
	m_Panf = panf;

	panf->user_data = this;
	PAN_SETDATA( panf, panel, this );
	panf->set(panel, PAN_USEROPEN, PanelUserOpenCallback);
	panf->set(panel, PAN_USERCLOSE, PanelUserCloseCallback);

	//*** Populate the tree
//	InitPopulation();

	//*** Build the controls
//	m_TreeControl = TREE_CTL( panf, panel, "Test", width, height, TreeInfo, TreeCount, TreeChild );
//	CON_SETEVENT( m_TreeControl, TreeEvent, this );

	m_MeshListBox = MULTILIST_CTL(panf, panel, "Mesh List", width, 5, MeshListBoxName, MeshListBoxCount, MeshListBoxColumnWidth);
	CON_SETEVENT( m_MeshListBox, MeshListBoxEvent, this );

	m_LWNameEdit = STRRO_CTL(panf, panel, "LW Name:",32);
	//CON_SETEVENT( m_LWNameEdit, LWNameEditEvent, this );

	m_DTSNameEdit = STR_CTL(panf, panel, "DTS Name:",32);
	pos = CON_Y( m_LWNameEdit );
	ival.intv.value=pos;
	m_DTSNameEdit->set(m_DTSNameEdit, CTL_Y, &ival);
	pos = CON_X( m_LWNameEdit );
	pos += CON_W( m_LWNameEdit );
	ival.intv.value=pos + 10;
	m_DTSNameEdit->set(m_DTSNameEdit, CTL_X, &ival);

	CON_SETEVENT( m_DTSNameEdit, DTSNameEditEvent, this );

	m_GlobalBorder0 = BORDER_CTL( panf, panel, "", (PAN_GETW( panf, panel ))-6, 0 );

	m_ExportTypePopup = POPUP_CTL( panf, panel, "Export As:", &(ExportTypePopupType[0]) );
	CON_SETEVENT( m_ExportTypePopup, ExportTypePopupEvent, this );

	m_GlobalRotationPopup = POPUP_CTL( panf, panel, "Rotate As:", &(GlobalRotationPopupType[0]) );
	CON_SETEVENT( m_GlobalRotationPopup, GlobalRotationPopupEvent, this );

	m_EncodedNormalsPopup = POPUP_CTL( panf, panel, "Polygon Normals:", &(EncodedNormalsPopupType[0]) );
	CON_SETEVENT( m_EncodedNormalsPopup, EncodedNormalsPopupEvent, this );

//	m_GlobalBorder0b = BORDER_CTL( panf, panel, "", (PAN_GETW( panf, panel ))-6, 0 );

	//*** Line up the Export and Rotation popups to the Encoded Normals Popup
	pos = CON_X( m_EncodedNormalsPopup );
	ival.intv.value=pos + 6;
	m_EncodedNormalsPopup->set(m_EncodedNormalsPopup, CTL_X, &ival);

	pos = CON_X( m_EncodedNormalsPopup );
	pos += CON_W( m_EncodedNormalsPopup );
	pos -= CON_W( m_ExportTypePopup );
	ival.intv.value=pos;
	m_ExportTypePopup->set(m_ExportTypePopup, CTL_X, &ival);

	pos = CON_X( m_EncodedNormalsPopup );
	pos += CON_W( m_EncodedNormalsPopup );
	pos -= CON_W( m_GlobalRotationPopup );
	ival.intv.value=pos;
	m_GlobalRotationPopup->set(m_GlobalRotationPopup, CTL_X, &ival);

	//*** Build the Tabs and the area for the tab controls to go
	m_GlobalTabChoice = TABCHOICE_CTL( panf, panel, "", &(GlobalTabChoices[0]) );
	ival.intv.value=10;
	m_GlobalTabChoice->set(m_GlobalTabChoice, CTL_X, &ival);
	pos = CON_Y( m_GlobalTabChoice );
	ival.intv.value=pos + 20;
	m_GlobalTabChoice->set(m_GlobalTabChoice, CTL_Y, &ival);
	CON_SETEVENT( m_GlobalTabChoice, GlobalTabChoiceEvent, this );

//	m_TabCanvas = BORDER_CTL( panf, panel, "", width, 200 );//AREA_CTL( panf, panel, "", 300, 200 );//CANVAS_CTL( panf, panel, "", 300, 200 );
//	pos = CON_Y( m_GlobalTabChoice );
//	pos += CON_H( m_GlobalTabChoice ) - 10;
//	ival.intv.value=pos;
//	m_TabCanvas->set(m_TabCanvas, CTL_Y, &ival);

	m_TabCanvas = m_GlobalTabChoice;

	//*** Build the Animation canvas and set their state accordingly
	m_AnimExportChoice = BOOL_CTL( panf, panel, "Export animation sequences" );
	CON_SETEVENT( m_AnimExportChoice, AnimExportChoiceEvent, this );
	pos = CON_Y( m_TabCanvas );
	pos += CON_H( m_TabCanvas );
	ival.intv.value=pos + 4;
	m_AnimExportChoice->set(m_AnimExportChoice, CTL_Y, &ival);
	pos = CON_X( m_TabCanvas );
	ival.intv.value=pos + 8;
	m_AnimExportChoice->set(m_AnimExportChoice, CTL_X, &ival);

	m_AnimIKSetupButton = BUTTON_CTL( panf, panel, "Setup for IK" );
	CON_SETEVENT( m_AnimIKSetupButton, AnimIKSetupButtonEvent, this );
	pos = CON_Y( m_AnimExportChoice );
	ival.intv.value=pos;
	m_AnimIKSetupButton->set(m_AnimIKSetupButton, CTL_Y, &ival);

	m_AnimBorder1 = BORDER_CTL( panf, panel, "", width, 0 );

	m_AnimNewButton = BUTTON_CTL( panf, panel, "New Seq" );
	CON_SETEVENT( m_AnimNewButton, AnimNewButtonEvent, this );
	pos = CON_X( m_TabCanvas );
	ival.intv.value=pos + 8;
	m_AnimNewButton->set(m_AnimNewButton, CTL_X, &ival);

	m_AnimDeleteButton = BUTTON_CTL( panf, panel, "Delete Seq" );
	CON_SETEVENT( m_AnimDeleteButton, AnimDeleteButtonEvent, this );
	pos = CON_Y( m_AnimNewButton );
	m_AnimDeleteButton->set(m_AnimDeleteButton, CTL_Y, &ival);
	pos = CON_X( m_AnimNewButton );
	pos += CON_W( m_AnimNewButton );
	ival.intv.value=pos + 8;
	m_AnimDeleteButton->set(m_AnimDeleteButton, CTL_X, &ival);

	m_AnimUpButton = BUTTON_CTL( panf, panel, "Up" );
	CON_SETEVENT( m_AnimUpButton, AnimUpButtonEvent, this );
	pos = CON_Y( m_AnimNewButton );
	m_AnimUpButton->set(m_AnimUpButton, CTL_Y, &ival);

	m_AnimDownButton = BUTTON_CTL( panf, panel, "Down" );
	CON_SETEVENT( m_AnimDownButton, AnimDownButtonEvent, this );
	pos = CON_Y( m_AnimNewButton );
	m_AnimDownButton->set(m_AnimDownButton, CTL_Y, &ival);

	m_AnimationListBox = MULTILIST_CTL(panf, panel, "Animation List", (width-32), 4, AnimListBoxName, AnimListBoxCount, AnimListBoxColumnWidth);
	CON_SETEVENT( m_AnimationListBox, AnimationListBoxEvent, this );
	pos = CON_X( m_TabCanvas );
	ival.intv.value=pos + 8;
	m_AnimationListBox->set(m_AnimationListBox, CTL_X, &ival);

	// Adjust the m_AnimDownButton and m_AnimUpButton controls x-pos based on m_AnimStartEdit
	pos = CON_X( m_AnimationListBox );
	pos += CON_W( m_AnimationListBox );
	pos -= CON_W( m_AnimDownButton );
	ival.intv.value=pos;
	m_AnimDownButton->set(m_AnimDownButton, CTL_X, &ival);
	pos = CON_X( m_AnimDownButton );
	pos -= CON_W( m_AnimUpButton );
	ival.intv.value=pos - 8;
	m_AnimUpButton->set(m_AnimUpButton, CTL_X, &ival);

	// Adjust the m_AnimIKSetupButton control's x-pos based on m_AnimStartEdit
	pos = CON_X( m_AnimationListBox );
	pos += CON_W( m_AnimationListBox );
	pos -= CON_W( m_AnimIKSetupButton );
	ival.intv.value=pos;
	m_AnimIKSetupButton->set(m_AnimIKSetupButton, CTL_X, &ival);


	m_AnimNameEdit = STR_CTL(panf, panel, "Seq Name:",32);
	CON_SETEVENT( m_AnimNameEdit, AnimNameEditEvent, this );
	pos = CON_X( m_TabCanvas );
	ival.intv.value=pos + 8;
	m_AnimNameEdit->set(m_AnimNameEdit, CTL_X, &ival);

	m_AnimFPSEdit = INT_CTL(panf, panel, "FPS:");
	CON_SETEVENT( m_AnimFPSEdit, AnimFPSEditEvent, this );
	pos = CON_Y( m_AnimNameEdit );
	m_AnimFPSEdit->set(m_AnimFPSEdit, CTL_Y, &ival);

	m_AnimStartEdit = INT_CTL(panf, panel, "Start Frame:");
	CON_SETEVENT( m_AnimStartEdit, AnimStartEditEvent, this );
	pos = CON_X( m_TabCanvas );
	ival.intv.value=pos + 8;
	m_AnimStartEdit->set(m_AnimStartEdit, CTL_X, &ival);

	// Adjust the m_AnimNameEdit and m_AnimFPSEdit controls x-pos based on m_AnimStartEdit
	pos = CON_LW(m_AnimStartEdit);
	pos -= CON_LW(m_AnimNameEdit);
	pos += CON_X( m_AnimStartEdit );
	ival.intv.value=pos;
	m_AnimNameEdit->set(m_AnimNameEdit, CTL_X, &ival);
	pos = CON_X( m_AnimNameEdit );
	pos += CON_W( m_AnimNameEdit );
	ival.intv.value=pos + 8;
	m_AnimFPSEdit->set(m_AnimFPSEdit, CTL_X, &ival);

	m_AnimEndEdit = INT_CTL(panf, panel, "End Frame:");
	CON_SETEVENT( m_AnimEndEdit, AnimEndEditEvent, this );
	pos = CON_Y( m_AnimStartEdit );
	m_AnimEndEdit->set(m_AnimEndEdit, CTL_Y, &ival);
	pos = CON_X( m_AnimFPSEdit );
	pos += CON_W( m_AnimFPSEdit );
	pos -= CON_W( m_AnimEndEdit );
	ival.intv.value=pos;
	m_AnimEndEdit->set(m_AnimEndEdit, CTL_X, &ival);

	m_AnimCyclicChoice = BOOL_CTL( panf, panel, "Sequence will automatically cycle" );
	CON_SETEVENT( m_AnimCyclicChoice, AnimCyclicChoiceEvent, this );
	pos = CON_X( m_TabCanvas );
	ival.intv.value=pos + 8;
	m_AnimCyclicChoice->set(m_AnimCyclicChoice, CTL_X, &ival);

	//*** Build the Global canvas and set their state accordingly
	m_SaveAs = SAVE_CTL( panf, panel, "Save As:", 50 );
	pos = CON_Y( m_TabCanvas );
	pos += CON_H( m_TabCanvas );
	ival.intv.value=pos + 4;
	m_SaveAs->set(m_SaveAs, CTL_Y, &ival);
	pos = CON_X( m_TabCanvas );
	ival.intv.value=pos + 8;
	m_SaveAs->set(m_SaveAs, CTL_X, &ival);
	CON_SETEVENT( m_SaveAs, SaveAsEvent, this );

	m_ExportMaterialsChoice = BOOL_CTL( panf, panel, "Export material information" );
	pos = CON_Y( m_SaveAs );
	pos += CON_H( m_SaveAs );
	ival.intv.value=pos;
	m_ExportMaterialsChoice->set(m_ExportMaterialsChoice, CTL_Y, &ival);
	pos = CON_X( m_TabCanvas );
	ival.intv.value=pos + 8;
	m_ExportMaterialsChoice->set(m_ExportMaterialsChoice, CTL_X, &ival);
	CON_SETEVENT( m_ExportMaterialsChoice, ExportMaterialsChoiceEvent, this );

	m_GlobalBorder1 = BORDER_CTL( panf, panel, "", width, 0 );
	pos = CON_Y( m_ExportMaterialsChoice );
	pos += CON_H( m_ExportMaterialsChoice );
	ival.intv.value=pos;
	m_GlobalBorder1->set(m_GlobalBorder1, CTL_Y, &ival);

	m_ScaleFactorEdit = FLOAT_CTL( panf, panel, "Global Scale Factor:" );
	pos = CON_Y( m_GlobalBorder1 );
	pos += CON_H( m_GlobalBorder1 );
	ival.intv.value=pos;
	m_ScaleFactorEdit->set(m_ScaleFactorEdit, CTL_Y, &ival);
	pos = CON_X( m_TabCanvas );
	ival.intv.value=pos + 8;
	m_ScaleFactorEdit->set(m_ScaleFactorEdit, CTL_X, &ival);
	CON_SETEVENT( m_ScaleFactorEdit, ScaleFactorEditEvent, this );

	m_MinimunVisibleSizeEdit = INT_CTL( panf, panel, "Minimun Visible Size (pixels):" );
	pos = CON_Y( m_ScaleFactorEdit );
	pos += CON_H( m_ScaleFactorEdit );
	ival.intv.value=pos;
	m_MinimunVisibleSizeEdit->set(m_MinimunVisibleSizeEdit, CTL_Y, &ival);
	pos = CON_X( m_TabCanvas );
	ival.intv.value=pos + 8;
	m_MinimunVisibleSizeEdit->set(m_MinimunVisibleSizeEdit, CTL_X, &ival);
	CON_SETEVENT( m_MinimunVisibleSizeEdit, MinimunVisibleSizeEditEvent, this );

		//*** Move ScaleFactorEdit to line up with MinimunVisibleSizeEdit
	pos = CON_X( m_MinimunVisibleSizeEdit );
	pos += CON_W( m_MinimunVisibleSizeEdit );
	pos -= CON_W( m_ScaleFactorEdit );
	ival.intv.value=pos;
	m_ScaleFactorEdit->set(m_ScaleFactorEdit, CTL_X, &ival);

	m_GlobalBorder2 = BORDER_CTL( panf, panel, "", width, 0 );
	pos = CON_Y( m_MinimunVisibleSizeEdit );
	pos += CON_H( m_MinimunVisibleSizeEdit );
	ival.intv.value=pos;
	m_GlobalBorder2->set(m_GlobalBorder2, CTL_Y, &ival);

	m_CollisionTypeChoice = POPUP_CTL( panf, panel, "Collision Mesh:", &(CollisionMeshChoices[0]) );
	pos = CON_Y( m_GlobalBorder2 );
	pos += CON_H( m_GlobalBorder2 );
	ival.intv.value=pos;
	m_CollisionTypeChoice->set(m_CollisionTypeChoice, CTL_Y, &ival);
	pos = CON_X( m_TabCanvas );
	ival.intv.value=pos + 8;
	m_CollisionTypeChoice->set(m_CollisionTypeChoice, CTL_X, &ival);
	CON_SETEVENT( m_CollisionTypeChoice, CollisionTypeChoiceEvent, this );

	m_CollisionVisibleChoice = BOOL_CTL( panf, panel, "Create a visible collision mesh" );
	pos = CON_Y( m_CollisionTypeChoice );
	pos += CON_H( m_CollisionTypeChoice );
	ival.intv.value=pos;
	m_CollisionVisibleChoice->set(m_CollisionVisibleChoice, CTL_Y, &ival);
	pos = CON_X( m_TabCanvas );
	ival.intv.value=pos + 8;
	m_CollisionVisibleChoice->set(m_CollisionVisibleChoice, CTL_X, &ival);
	CON_SETEVENT( m_CollisionVisibleChoice, CollisionVisibleChoiceEvent, this );

	m_CollisionComplexityEdit = FLOAT_CTL( panf, panel, "Collision Complexity:" );
	pos = CON_Y( m_CollisionVisibleChoice );
	pos += CON_H( m_CollisionVisibleChoice );
	ival.intv.value=pos;
	m_CollisionComplexityEdit->set(m_CollisionComplexityEdit, CTL_Y, &ival);
	pos = CON_X( m_TabCanvas );
	ival.intv.value=pos + 8;
	m_CollisionComplexityEdit->set(m_CollisionComplexityEdit, CTL_X, &ival);
	CON_SETEVENT( m_CollisionComplexityEdit, CollisionComplexityEditEvent, this );

	//*** Build the Advanced canvas and set their state accordingly
	m_PolyPointTypeChoice = POPUP_CTL( panf, panel, "Point and Poly Type:", &(PolyPointTypeChoices[0]) );
	pos = CON_Y( m_TabCanvas );
	pos += CON_H( m_TabCanvas );
	ival.intv.value=pos + 4;
	m_PolyPointTypeChoice->set(m_PolyPointTypeChoice, CTL_Y, &ival);
	pos = CON_X( m_TabCanvas );
	ival.intv.value=pos + 8;
	m_PolyPointTypeChoice->set(m_PolyPointTypeChoice, CTL_X, &ival);
	CON_SETEVENT( m_PolyPointTypeChoice, PolyPointTypeChoiceEvent, this );

	m_BaseTransFrameEdit = FLOAT_CTL( panf, panel, "Initial transforms based on time (sec):" );
	pos = CON_Y( m_PolyPointTypeChoice );
	pos += CON_H( m_PolyPointTypeChoice );
	ival.intv.value=pos;
	m_BaseTransFrameEdit->set(m_BaseTransFrameEdit, CTL_Y, &ival);
	pos = CON_X( m_TabCanvas );
	ival.intv.value=pos + 8;
	m_BaseTransFrameEdit->set(m_BaseTransFrameEdit, CTL_X, &ival);
	CON_SETEVENT( m_BaseTransFrameEdit, BaseTransFrameEditEvent, this );

	m_ExportMeshAsNodeChoice = BOOL_CTL( panf, panel, "Export meshes as nodes" );
	pos = CON_Y( m_BaseTransFrameEdit );
	pos += CON_H( m_BaseTransFrameEdit );
	ival.intv.value=pos;
	m_ExportMeshAsNodeChoice->set(m_ExportMeshAsNodeChoice, CTL_Y, &ival);
	pos = CON_X( m_TabCanvas );
	ival.intv.value=pos + 8;
	m_ExportMeshAsNodeChoice->set(m_ExportMeshAsNodeChoice, CTL_X, &ival);
	CON_SETEVENT( m_ExportMeshAsNodeChoice, ExportMeshAsNodeChoiceEvent, this );

	m_MeshNodeExtEdit = STR_CTL(panf, panel, "Mesh extension for nodes:",32);
	pos = CON_Y( m_ExportMeshAsNodeChoice );
	pos += CON_H( m_ExportMeshAsNodeChoice );
	ival.intv.value=pos;
	m_MeshNodeExtEdit->set(m_MeshNodeExtEdit, CTL_Y, &ival);
	pos = CON_X( m_TabCanvas );
	ival.intv.value=pos + 8;
	m_MeshNodeExtEdit->set(m_MeshNodeExtEdit, CTL_X, &ival);
	CON_SETEVENT( m_MeshNodeExtEdit, MeshNodeExtEditEvent, this );

	//*** The Export Button
	m_ExportButton = BUTTON_CTL( panf, panel, "Export DTS File" );
	CON_SETEVENT( m_ExportButton, ExportButtonEvent, this );

	//*** The Close button
	m_CloseButton = BUTTON_CTL( panf, panel, "Close" );
	CON_SETEVENT( m_CloseButton, CloseButtonEvent, this );

	//*** The View Nodes button
	m_ViewNodesButton = BUTTON_CTL( panf, panel, "View Nodes" );
	CON_SETEVENT( m_ViewNodesButton, ViewNodesButtonEvent, this );

	//*** Move the Export button
	pos = CON_Y( m_TabCanvas );
	pos += CON_H( m_TabCanvas );
	ival.intv.value=pos + 252;
	m_ExportButton->set(m_ExportButton, CTL_Y, &ival);

	//*** Move the close button
	pos = CON_Y( m_ExportButton );
	ival.intv.value=pos;
	m_CloseButton->set(m_CloseButton, CTL_Y, &ival);
	pos = PAN_GETW( panf, panel );
	pos -= CON_W( m_CloseButton );
	ival.intv.value=pos;
	m_CloseButton->set(m_CloseButton, CTL_X, &ival);

	//*** Move the View Nodes button
	pos = CON_Y( m_ExportButton );
	ival.intv.value=pos;
	m_ViewNodesButton->set(m_ViewNodesButton, CTL_Y, &ival);
	pos = PAN_GETW( panf, panel );
	pos -= CON_W( m_CloseButton );
	pos -= CON_W( m_ViewNodesButton );
	ival.intv.value=pos - 10;
	m_ViewNodesButton->set(m_ViewNodesButton, CTL_X, &ival);

	//*** Set the panel draw callback
	PAN_SETDRAW( panf, panel, PanelDrawCallback );

	//*** Set up values of controls
//	SET_INT( m_MeshListBox, m_InstData->m_nLastSelected );
	SET_STR( m_LWNameEdit, "", 0);
	SET_STR( m_DTSNameEdit, "", 0);
	SET_STR( m_SaveAs, m_InstData->m_szFilename, strlen(m_InstData->m_szFilename));
	SET_INT( m_ExportMaterialsChoice, m_InstData->m_nExportMaterials );
	SET_INT( m_CollisionTypeChoice, m_InstData->m_nCollisionType );
	SET_INT( m_CollisionVisibleChoice, m_InstData->m_nCollisionVisible );
	SET_INT( m_MinimunVisibleSizeEdit, m_InstData->m_nMinPixelSize );
	SET_FLOAT( m_ScaleFactorEdit, m_InstData->m_fScaleFactor );
	SET_FLOAT( m_CollisionComplexityEdit, m_InstData->m_fCollisionComplexity );

	SET_INT( m_AnimExportChoice, m_InstData->m_nExportAnimSequences );

	SET_INT( m_PolyPointTypeChoice, m_InstData->m_nPolyPointType );
	SET_FLOAT( m_BaseTransFrameEdit, m_InstData->m_fBaseTransFrame );
	SET_INT( m_ExportMeshAsNodeChoice, m_InstData->m_nExportMeshAsNode );
	SET_STR( m_MeshNodeExtEdit, m_InstData->m_szMeshNodeExt, strlen(m_InstData->m_szMeshNodeExt));

	//*** Set the Global canvas to render and the rest to not
	SetAnimationCanvas(DR_ERASE);
	SetAdvancedCanvas(DR_ERASE);
	SetGlobalCanvas(DR_RENDER);


	return true;
}

//*** Set the drawing mode of the Global canvas and refresh
void CGUIMeshSelector::SetGlobalCanvas(DrMode mode)
{
	m_SaveAs->draw(m_SaveAs, mode);
	m_ScaleFactorEdit->draw(m_ScaleFactorEdit, mode);
	m_MinimunVisibleSizeEdit->draw(m_MinimunVisibleSizeEdit, mode);
	m_CollisionTypeChoice->draw(m_CollisionTypeChoice, mode);
	m_CollisionVisibleChoice->draw(m_CollisionVisibleChoice, mode);
	m_CollisionComplexityEdit->draw(m_CollisionComplexityEdit, mode);
	m_ExportMaterialsChoice->draw(m_ExportMaterialsChoice, mode);
	m_GlobalBorder1->draw(m_GlobalBorder1, mode);
	m_GlobalBorder2->draw(m_GlobalBorder2, mode);

}

//*** Set the drawing mode of the Animation canvas and refresh
void CGUIMeshSelector::SetAnimationCanvas(DrMode mode)
{
	m_AnimExportChoice->draw(m_AnimExportChoice, mode);
	m_AnimIKSetupButton->draw(m_AnimIKSetupButton, mode);
	m_AnimationListBox->draw(m_AnimationListBox, mode);
	m_AnimNewButton->draw(m_AnimNewButton, mode);
	m_AnimDeleteButton->draw(m_AnimDeleteButton, mode);
	m_AnimBorder1->draw(m_AnimBorder1, mode);
	m_AnimNameEdit->draw(m_AnimNameEdit, mode);
	m_AnimFPSEdit->draw(m_AnimFPSEdit, mode);
	m_AnimStartEdit->draw(m_AnimStartEdit, mode);
	m_AnimEndEdit->draw(m_AnimEndEdit, mode);
	m_AnimCyclicChoice->draw(m_AnimCyclicChoice, mode);

	//*** Handle the Up and Down buttons
	if(mode == DR_RENDER)
	{
		//*** Handle the Up & Down buttons
		if(m_InstData->GetAnimSequenceCount() > 1)
		{
			m_AnimUpButton->draw(m_AnimUpButton, mode);
			m_AnimDownButton->draw(m_AnimDownButton, mode);
		} else
		{
			m_AnimUpButton->draw(m_AnimUpButton, mode);
			m_AnimDownButton->draw(m_AnimDownButton, mode);
			m_AnimUpButton->draw(m_AnimUpButton, DR_GHOST);
			m_AnimDownButton->draw(m_AnimDownButton, DR_GHOST);
		}

		//*** Handle the Delete Seq button
		if(m_nAnimSelectedRow > -1)
		{
			m_AnimDeleteButton->draw(m_AnimDeleteButton, mode);
		} else
		{
			m_AnimDeleteButton->draw(m_AnimDeleteButton, mode);
			m_AnimDeleteButton->draw(m_AnimDeleteButton, DR_GHOST);
		}
	} else
	{
		m_AnimUpButton->draw(m_AnimUpButton, mode);
		m_AnimDownButton->draw(m_AnimDownButton, mode);
		m_AnimDeleteButton->draw(m_AnimDeleteButton, mode);
	}
}

//*** Set the drawing mode of the Advanced canvas and refresh
void CGUIMeshSelector::SetAdvancedCanvas(DrMode mode)
{
	m_PolyPointTypeChoice->draw(m_PolyPointTypeChoice, mode);
	m_BaseTransFrameEdit->draw(m_BaseTransFrameEdit, mode);
	m_ExportMeshAsNodeChoice->draw(m_ExportMeshAsNodeChoice, mode);
	m_MeshNodeExtEdit->draw(m_MeshNodeExtEdit, mode);
}

//*** Refresh the animation sequence list
void CGUIMeshSelector::RefreshAnimList()
{
	m_AnimationListBox->draw(m_AnimationListBox, DR_REFRESH);
}

//*** Refresh the mesh list
void CGUIMeshSelector::RefreshMeshList()
{
	m_MeshListBox->draw(m_MeshListBox, DR_REFRESH);
}

//*** Initalize the population of the mesh list
bool CGUIMeshSelector::InitPopulation()
{
	LWItemID id;

	id = CLWGlobal::iteminfo->first( LWI_OBJECT, NULL );
	while(id != LWITEM_NULL)
	{
		ProcessObject(id);

		id = CLWGlobal::iteminfo->next(id);
	}

	return true;
}

//*** Process an object
void CGUIMeshSelector::ProcessObject(LWItemID id)
{
/*
	//*** Build out a node with the object's data
	MeshSelectionNode* node = new MeshSelectionNode;
	node->m_ID = id;
	strncpy(node->m_szName,CLWGlobal::iteminfo->name(id), MAX_NODE_NAME);
	node->m_nChildren = 0;
	node->m_nFlags = 0;
	node->m_nExportType = 0;
	strcpy(node->m_szDTSName,"");

	//*** Add the node to the item list
	m_ItemList.push_back(node);
*/
}


//************************************************************
//*** Functions for the bottom buttons and tabs
//************************************************************

//*** Functions for the export mesh button
static void ExportButtonEvent( LWControl *ectl, void *edata )
{
	LWPanControlDesc desc;
	LWValue
		ival    = { LWT_INTEGER },
		ivecval = { LWT_VINT },
		fval    = { LWT_FLOAT },
		fvecval = { LWT_VFLOAT },
		sval    = { LWT_STRING };

	CGUIMeshSelector* data = (CGUIMeshSelector*)edata;

//	CGUIMeshSelector* instdata = (CGUIMeshSelector*) edata;
	int myvalue;

	GET_INT(data->m_MeshListBox, myvalue);

//	char buf[128];
//	sprintf(buf,"%i",myvalue);
//	::MessageBox(NULL,buf,"Info",MB_OK);

	//*** Check all user parameters are correct and we may export
	if(data->m_InstData->CheckParameters() == false)
	{
		return;
	}

	std::ofstream file (data->m_InstData->m_szFilename, std::ios::binary | std::ios::trunc | std::ios::out) ;

	if (!file)
	{
		CLWGlobal::msg->error("Error openning DTS file for writing.",data->m_InstData->m_szFilename);
	} else
	{
		//*** Initialize a Layout Monitor
		int numobj = CLWGlobal::GetObjectCount();
		CLWGlobal::LayoutMonitorInit("DTS file export", (LMO_REVIEW|LMO_IMMUPD), NULL, (numobj + 5), "Starting...");

		DTS::CLWShape::ImportConfig config;
		DTS::CLWShape shape(data->m_InstData) ;
		CLWGlobal::LayoutMonitorStep(1, "Saving DTS file...");
		shape.save(file) ;

		file.close();

//		CLWGlobal::msg->info("DTS file save complete.","");
		CLWGlobal::LayoutMonitorDone();

	}
}

//*** The Close button
static void CloseButtonEvent( LWControl *ectl, void *edata )
{
	LWPanControlDesc desc;
	LWValue
		ival    = { LWT_INTEGER },
		ivecval = { LWT_VINT },
		fval    = { LWT_FLOAT },
		fvecval = { LWT_VFLOAT },
		sval    = { LWT_STRING };

	CGUIMeshSelector* data = (CGUIMeshSelector*)edata;

	data->m_Panf->close(data->m_Panel);
}

//*** Functions for the View Node button
static void ViewNodesButtonEvent( LWControl *ectl, void *edata )
{
	LWPanControlDesc desc;
	LWValue
		ival    = { LWT_INTEGER },
		ivecval = { LWT_VINT },
		fval    = { LWT_FLOAT },
		fvecval = { LWT_VFLOAT },
		sval    = { LWT_STRING };

	CGUIMeshSelector* data = (CGUIMeshSelector*)edata;
	LWPanelID panel = 0;
	LWPanelFuncs *panf = (LWPanelFuncs *) (CLWGlobal::global)( LWPANELFUNCS_GLOBAL, GFUSE_TRANSIENT );
	CGUIViewNodes	nodelist;

	//*** Define the panel
	if( !( panel = PAN_CREATE( panf, "Node List" )))
	{
		CLWGlobal::msg->info("Cannot open the Node List interface!","");
		return;
	}

	//*** Open the panel
	if(panel)
	{
		//*** Create all of the controls
		nodelist.CreateNodeList(data->m_InstData, panf, panel, 0, 0, 500, 200);

		//*** Display the panel
		panf->open( panel, PANF_BLOCKING );

		PAN_KILL( panf, panel );
	}

}

//*** Event for the GlobalTabChoice control
static void GlobalTabChoiceEvent( LWControl *ectl, void *edata )
{
	LWPanControlDesc desc;
	LWValue
		ival    = { LWT_INTEGER },
		ivecval = { LWT_VINT },
		fval    = { LWT_FLOAT },
		fvecval = { LWT_VFLOAT },
		sval    = { LWT_STRING };

	int value;
 	CGUIMeshSelector* data = (CGUIMeshSelector*)edata;

	GET_INT( ectl, value);

	switch(value)
	{
	case 0:
		data->SetAnimationCanvas(DR_ERASE);
		data->SetAdvancedCanvas(DR_ERASE);
		data->SetGlobalCanvas(DR_RENDER);
		break;

	case 1:
		data->SetGlobalCanvas(DR_ERASE);
		data->SetAdvancedCanvas(DR_ERASE);
		data->SetAnimationCanvas(DR_RENDER);
		break;

	case 2:
		data->SetGlobalCanvas(DR_ERASE);
		data->SetAnimationCanvas(DR_ERASE);
		data->SetAdvancedCanvas(DR_RENDER);
		break;

	default:
		break;
	}

}


//************************************************************
//*** Functions for the upper mesh section
//************************************************************

//*** Returns the number of rows for the Mesh List Box
static int MeshListBoxCount( void *userdata )
{
	//::MessageBox(NULL,"MeshListBoxCount","Info",MB_OK);

	if(userdata)
	{
	 	CGUIMeshSelector* data = (CGUIMeshSelector*)userdata;

		return (data->m_InstData->GetObjectCount());

	} else
	{
		return 0;
	}
}

//*** Returns the text for a cell [row,column = index,column] for the Mesh List Box
static char* MeshListBoxName( void *userdata, int index, int column )
{
	//char tempbuf[1024];
	//sprintf(tempbuf,"MeshListBoxName Index %i Column %i",index,column);
	//::MessageBox(NULL,tempbuf,"Info",MB_OK);

	if(userdata)
	{
 		CGUIMeshSelector* data = (CGUIMeshSelector*)userdata;

		if(column < MeshMaxColumns)
		{
			switch(column)
			{
			case 0:
				//*** Ex Column
				if(index == -1)
				{
					//*** Return the column title
					return MeshLBColumnHeadings[column];

				} else if(index >= 0 && index < data->m_InstData->GetObjectCount() )
				{
					return MeshLBExportType[data->m_InstData->m_vObjectList[index]->m_nExportType];
				}
				return NULL;
				break;

			case 1:
				//*** LW Name column
				if(index == -1)
				{
					//*** Return the column title
					return MeshLBColumnHeadings[column];

				} else if(index >= 0 && index < data->m_InstData->GetObjectCount() )
				{
					return data->m_InstData->m_vObjectList[index]->m_szLWName;
				}
				return NULL;
				break;

			case 2:
				//*** DTS Name column
				if(index == -1)
				{
					//*** Return the column title
					return MeshLBColumnHeadings[column];

				} else if(index >= 0 && index < data->m_InstData->GetObjectCount() )
				{
					if(data->m_InstData->m_vObjectList[index]->m_szDTSName[0] == '\0')
					{
						return data->m_InstData->m_vObjectList[index]->m_szLWName;
					} else
					{
						return data->m_InstData->m_vObjectList[index]->m_szDTSName;
					}
				}
				return NULL;
				break;

			case 3:
				//*** Rotation column
				if(index == -1)
				{
					//*** Return the column title
					return MeshLBColumnHeadings[column];

				} else if(index >= 0 && index < data->m_InstData->GetObjectCount() )
				{
					return MeshLBRotation[data->m_InstData->m_vObjectList[index]->m_nGlobalRotation];
				}
				return NULL;
				break;

			case 4:
				//*** Encoded Normals column
				if(index == -1)
				{
					//*** Return the column title
					return MeshLBColumnHeadings[column];

				} else if(index >= 0 && index < data->m_InstData->GetObjectCount() )
				{
					return MeshLBEncodedNormals[data->m_InstData->m_vObjectList[index]->m_nEncodedNormals];
				}
				return NULL;
				break;

			default:
				return NULL;
				break;
			}

		} else
		{
			return NULL;
		}
	} else
	{
		return NULL;
	}
}

//*** Return the width of a given column for the Mesh List Box
static int MeshListBoxColumnWidth( void *userdata, int column )
{
	//char tempbuf[1024];
	//sprintf(tempbuf,"MeshListBoxColumnWidth Column %i",column);
	//::MessageBox(NULL,tempbuf,"Info",MB_OK);

	if(userdata)
	{
 		CGUIMeshSelector* data = (CGUIMeshSelector*)userdata;

		if(column < MeshMaxColumns)
		{
			//*** If Layout is asking for a column we want defined...
			return MeshLBColumnWidths[column];
		} else
		{
			//*** ...otherwise return 0 to indicate a column we don't need
			return 0;
		}

	} else
	{
		return 0;
	}
}
//*** Event callback for the Mesh List Box
static void MeshListBoxEvent( LWControl *ectl, void *edata )
{
	LWPanControlDesc desc;
	LWValue
		ival    = { LWT_INTEGER },
		ivecval = { LWT_VINT },
		fval    = { LWT_FLOAT },
		fvecval = { LWT_VFLOAT },
		sval    = { LWT_STRING };

	CGUIMeshSelector* data = (CGUIMeshSelector*)edata;
	int sel;

	//*** Retrieve the selected row from the list box and store it
	ectl->get( ectl, CTL_VALUE, &ival );
	sel = (int) ival.intv.value;
	data->m_InstData->m_nLastSelected = sel;
	data->m_nSelectedRow = sel;

	//*** Change the text of the edit controls to match the selection
	SET_STR( data->m_LWNameEdit, (data->m_InstData->m_vObjectList[sel]->m_szLWName), strlen(data->m_InstData->m_vObjectList[sel]->m_szLWName));
	SET_STR( data->m_DTSNameEdit, (data->m_InstData->m_vObjectList[sel]->m_szDTSName), strlen(data->m_InstData->m_vObjectList[sel]->m_szDTSName));

	//*** Change the Export Type Popup
	SET_INT( data->m_ExportTypePopup, (data->m_InstData->m_vObjectList[sel]->m_nExportType) );

	//*** Change the Rotation Popup
	SET_INT( data->m_GlobalRotationPopup, (data->m_InstData->m_vObjectList[sel]->m_nGlobalRotation) );

	//*** Change the Polygon Normal Popup
	SET_INT( data->m_EncodedNormalsPopup, (data->m_InstData->m_vObjectList[sel]->m_nEncodedNormals) );
}

//*** Event for the DTSNameEdit control
static void DTSNameEditEvent( LWControl *ectl, void *edata )
{
	LWPanControlDesc desc;
	LWValue
		ival    = { LWT_INTEGER },
		ivecval = { LWT_VINT },
		fval    = { LWT_FLOAT },
		fvecval = { LWT_VFLOAT },
		sval    = { LWT_STRING };

	char strbuf[MAX_NODE_NAME];
 	CGUIMeshSelector* data = (CGUIMeshSelector*)edata;

	GET_STR( ectl, strbuf, MAX_NODE_NAME);

	//*** Store the new name into the row if a row has been selected
	if(data->m_nSelectedRow > -1)
	{
		strncpy(data->m_InstData->m_vObjectList[data->m_nSelectedRow]->m_szDTSName,strbuf,MAX_NODE_NAME);
		data->RefreshMeshList();
	}

}

//*** Event for the ExportTypePopup control
static void ExportTypePopupEvent( LWControl *ectl, void *edata )
{
	LWPanControlDesc desc;
	LWValue
		ival    = { LWT_INTEGER },
		ivecval = { LWT_VINT },
		fval    = { LWT_FLOAT },
		fvecval = { LWT_VFLOAT },
		sval    = { LWT_STRING };

	char value;
 	CGUIMeshSelector* data = (CGUIMeshSelector*)edata;

	GET_INT( ectl, value);

	//*** Store the new name into the row if a row has been selected
	if(data->m_nSelectedRow > -1)
	{
		data->m_InstData->m_vObjectList[data->m_nSelectedRow]->m_nExportType = value;
		data->RefreshMeshList();
	}

}

//*** Event for the GlobalRotationPopupPopup control
static void GlobalRotationPopupEvent( LWControl *ectl, void *edata )
{
	LWPanControlDesc desc;
	LWValue
		ival    = { LWT_INTEGER },
		ivecval = { LWT_VINT },
		fval    = { LWT_FLOAT },
		fvecval = { LWT_VFLOAT },
		sval    = { LWT_STRING };

	char value;
 	CGUIMeshSelector* data = (CGUIMeshSelector*)edata;

	GET_INT( ectl, value);

	//*** Store the new name into the row if a row has been selected
	if(data->m_nSelectedRow > -1)
	{
		data->m_InstData->m_vObjectList[data->m_nSelectedRow]->m_nGlobalRotation = value;
		data->RefreshMeshList();
	}

}

//*** Event for the EncodedNormalsPopupPopup control
static void EncodedNormalsPopupEvent( LWControl *ectl, void *edata )
{
	LWPanControlDesc desc;
	LWValue
		ival    = { LWT_INTEGER },
		ivecval = { LWT_VINT },
		fval    = { LWT_FLOAT },
		fvecval = { LWT_VFLOAT },
		sval    = { LWT_STRING };

	char value;
 	CGUIMeshSelector* data = (CGUIMeshSelector*)edata;

	GET_INT( ectl, value);

	//*** Store the new name into the row if a row has been selected
	if(data->m_nSelectedRow > -1)
	{
		data->m_InstData->m_vObjectList[data->m_nSelectedRow]->m_nEncodedNormals = value;
		data->RefreshMeshList();
	}

}


//************************************************************
//*** Functions for the Global Tab
//************************************************************

//*** Event for the SaveAs control
static void SaveAsEvent( LWControl *ectl, void *edata )
{
	LWPanControlDesc desc;
	LWValue ival    = { LWT_INTEGER },ivecval = { LWT_VINT },fval    = { LWT_FLOAT },fvecval = { LWT_VFLOAT },sval    = { LWT_STRING };

	CGUIMeshSelector* data = (CGUIMeshSelector*)edata;
	
	GET_STR(ectl, data->m_InstData->m_szFilename, MAX_FILE_PATH );
}

//*** Event for the ExportMaterialsChoice control
static void ExportMaterialsChoiceEvent( LWControl *ectl, void *edata )
{
	LWPanControlDesc desc;
	LWValue ival    = { LWT_INTEGER },ivecval = { LWT_VINT },fval    = { LWT_FLOAT },fvecval = { LWT_VFLOAT },sval    = { LWT_STRING };

	CGUIMeshSelector* data = (CGUIMeshSelector*)edata;
	int value;

	GET_INT(ectl, value);
	data->m_InstData->m_nExportMaterials = value;
}

//*** Event for the ScaleFactorEdit control
static void ScaleFactorEditEvent( LWControl *ectl, void *edata )
{
	LWPanControlDesc desc;
	LWValue ival    = { LWT_INTEGER },ivecval = { LWT_VINT },fval    = { LWT_FLOAT },fvecval = { LWT_VFLOAT },sval    = { LWT_STRING };

	CGUIMeshSelector* data = (CGUIMeshSelector*)edata;
	float value;

	GET_FLOAT(ectl, value);
	data->m_InstData->m_fScaleFactor = value;
}

//*** Event for the MinimunVisibleSizeEdit control
static void MinimunVisibleSizeEditEvent( LWControl *ectl, void *edata )
{
	LWPanControlDesc desc;
	LWValue ival    = { LWT_INTEGER },ivecval = { LWT_VINT },fval    = { LWT_FLOAT },fvecval = { LWT_VFLOAT },sval    = { LWT_STRING };

	CGUIMeshSelector* data = (CGUIMeshSelector*)edata;
	int value;

	GET_INT(ectl, value);
	data->m_InstData->m_nMinPixelSize = value;
}

//*** Event for CollisionTypeChoice control
static void CollisionTypeChoiceEvent( LWControl *ectl, void *edata )
{
	LWPanControlDesc desc;
	LWValue ival    = { LWT_INTEGER },ivecval = { LWT_VINT },fval    = { LWT_FLOAT },fvecval = { LWT_VFLOAT },sval    = { LWT_STRING };

	CGUIMeshSelector* data = (CGUIMeshSelector*)edata;
	int value;

	GET_INT(ectl, value);
	data->m_InstData->m_nCollisionType = value;
}

//*** Event for CollisionVisibleChoice control
static void CollisionVisibleChoiceEvent( LWControl *ectl, void *edata )
{
	LWPanControlDesc desc;
	LWValue ival    = { LWT_INTEGER },ivecval = { LWT_VINT },fval    = { LWT_FLOAT },fvecval = { LWT_VFLOAT },sval    = { LWT_STRING };

	CGUIMeshSelector* data = (CGUIMeshSelector*)edata;
	int value;

	GET_INT(ectl, value);
	data->m_InstData->m_nCollisionVisible = value;
}

//*** Event for the CollisionComplexityEdit control
static void CollisionComplexityEditEvent( LWControl *ectl, void *edata )
{
	LWPanControlDesc desc;
	LWValue ival    = { LWT_INTEGER },ivecval = { LWT_VINT },fval    = { LWT_FLOAT },fvecval = { LWT_VFLOAT },sval    = { LWT_STRING };

	CGUIMeshSelector* data = (CGUIMeshSelector*)edata;
	float value;

	GET_FLOAT(ectl, value);
	data->m_InstData->m_fCollisionComplexity = value;
}


//************************************************************
//*** Functions for the Animation Tab
//************************************************************

//*** Event for the m_AnimExportChoice control
static void AnimExportChoiceEvent( LWControl *ectl, void *edata )
{
	LWPanControlDesc desc;
	LWValue ival = { LWT_INTEGER },ivecval = { LWT_VINT },fval = { LWT_FLOAT },fvecval = { LWT_VFLOAT },sval    = { LWT_STRING };

	CGUIMeshSelector* data = (CGUIMeshSelector*)edata;
	int value;

	GET_INT(ectl, value);
	data->m_InstData->m_nExportAnimSequences = value;
}

//*** Event for the m_AnimIKSetupButton control
static void AnimIKSetupButtonEvent( LWControl *ectl, void *edata )
{
	LWPanControlDesc desc;
	LWValue	ival    = { LWT_INTEGER }, ivecval = { LWT_VINT }, fval    = { LWT_FLOAT }, fvecval = { LWT_VFLOAT }, sval    = { LWT_STRING };

	CGUIMeshSelector* data = (CGUIMeshSelector*)edata;

	//*** Go through the objects and setup the special IK plug-in as required
	int numobj = CLWGlobal::GetObjectCount();
	CLWGlobal::LayoutMonitorInit("IK Setup for DTS Exporter", (LMO_REVIEW|LMO_IMMUPD), NULL, (numobj), "Starting...");

	LWItemID id;
	id = CLWGlobal::iteminfo->first( LWI_OBJECT, NULL );
	while ( id )
	{
		char buffer[256];

		sprintf(buffer,"Working on object %s",CLWGlobal::iteminfo->name(id));
		CLWGlobal::LayoutMonitorStep(1, buffer);

		//*** Check if the object is controlled by IK
		int motType[3];
		CLWGlobal::iteminfo->controller( id, LWIP_ROTATION, motType );
		if( (motType[0] == LWMOTCTL_IK) || (motType[1] == LWMOTCTL_IK) || (motType[2] == LWMOTCTL_IK) )
		{
			//*** This bone is controlled by IK.
			CLWGlobal::LayoutMonitorStep(0, "---- Object is controlled by IK...Adding motion plug-in");

			//*** Select the bone
			sprintf(buffer,"SelectItem %x",id);
			CLWGlobal::evaluate(buffer);

			//*** Apply the plug-in
			sprintf(buffer,"ApplyServer %s %s",LWITEMMOTION_HCLASS,"TorqueDTS_IKChannelCreator");
			CLWGlobal::evaluate(buffer);
		} else
		{
		}

		LWItemID boneid;
		boneid = CLWGlobal::iteminfo->first( LWI_BONE, id );
		while( boneid )
		{
			sprintf(buffer,"-- Working on bone %s",CLWGlobal::iteminfo->name(boneid));
			CLWGlobal::LayoutMonitorStep(0, buffer);

			//*** Check if this bone has IK activated on any channel
			int motType[3];
			CLWGlobal::iteminfo->controller( boneid, LWIP_ROTATION, motType );
			if( (motType[0] == LWMOTCTL_IK) || (motType[1] == LWMOTCTL_IK) || (motType[2] == LWMOTCTL_IK) )
			{
				//*** This bone is controlled by IK.
				CLWGlobal::LayoutMonitorStep(0, "---- Bone is controlled by IK...Adding motion plug-in");

				//*** Select the bone
				sprintf(buffer,"SelectItem %x",boneid);
				CLWGlobal::evaluate(buffer);

				//*** Apply the plug-in
				sprintf(buffer,"ApplyServer %s %s",LWITEMMOTION_HCLASS,"TorqueDTS_IKChannelCreator");
				CLWGlobal::evaluate(buffer);
			} else
			{
				//*** IK not activated for this bone.  Remove plug-in and channels as needed
				int index = 1;
				const char* servname;

				servname = CLWGlobal::iteminfo->server( boneid, LWITEMMOTION_HCLASS, index );
				while(servname != NULL)
				{
					if(!strcmp(servname, "TorqueDTS_IKChannelCreator"))
					{
						//*** Found the plug-in, so remove it and the channels
						CLWGlobal::LayoutMonitorStep(0, "---- Bone is no longer controlled by IK...Removing motion plug-in & channels");

						//*** Select the bone
						sprintf(buffer,"SelectItem %x",boneid);
						CLWGlobal::evaluate(buffer);

						//*** Remove the plug-in
						sprintf(buffer,"RemoveServer %s %i",LWITEMMOTION_HCLASS,index);
						CLWGlobal::evaluate(buffer);

						//*** Now go through the channels to remove the DTS specific ones
						LWChanGroupID	chanGroup;
						LWEnvelopeID envid;
						chanGroup = CLWGlobal::iteminfo->chanGroup(boneid);

						envid = CLWGlobal::findEnv( chanGroup, "DTSIK.H" );
						if(envid)
						{
							CLWGlobal::envfunc->destroy(envid);
						}

						envid = CLWGlobal::findEnv( chanGroup, "DTSIK.P" );
						if(envid)
						{
							CLWGlobal::envfunc->destroy(envid);
						}

						envid = CLWGlobal::findEnv( chanGroup, "DTSIK.B" );
						if(envid)
						{
							CLWGlobal::envfunc->destroy(envid);
						}


					}
					++index;
					servname = CLWGlobal::iteminfo->server( boneid, LWITEMMOTION_HCLASS, index );
				}
			}

			boneid = CLWGlobal::iteminfo->next( boneid );
		}

		id = CLWGlobal::iteminfo->next( id );
	}

	CLWGlobal::LayoutMonitorDone();

}

//*** Returns the number of rows for the Anim list box
static int AnimListBoxCount( void *userdata )
{
	//::MessageBox(NULL,"MeshListBoxCount","Info",MB_OK);

	if(userdata)
	{
	 	CGUIMeshSelector* data = (CGUIMeshSelector*)userdata;

		return (data->m_InstData->GetAnimSequenceCount());

	} else
	{
		return 0;
	}
}

//*** Returns the text for a cell [row,column = index,column] for the Anim list box
static char* AnimListBoxName( void *userdata, int index, int column )
{
	if(userdata)
	{
 		CGUIMeshSelector* data = (CGUIMeshSelector*)userdata;

		if(column < AnimMaxColumns)
		{
			switch(column)
			{
			case 0:
				//*** Name Column
				if(index == -1)
				{
					//*** Return the column title
					return AnimLBColumnHeadings[column];

				} else if(index >= 0 && index < data->m_InstData->GetAnimSequenceCount() )
				{
					return data->m_InstData->m_vSequences[index]->m_szName;
				}
				return NULL;
				break;

			case 1:
				//*** Start column
				if(index == -1)
				{
					//*** Return the column title
					return AnimLBColumnHeadings[column];

				} else if(index >= 0 && index < data->m_InstData->GetAnimSequenceCount() )
				{
					return data->m_InstData->m_vSequences[index]->m_szStart;
				}
				return NULL;
				break;

			case 2:
				//*** End column
				if(index == -1)
				{
					//*** Return the column title
					return AnimLBColumnHeadings[column];

				} else if(index >= 0 && index < data->m_InstData->GetAnimSequenceCount() )
				{
					return data->m_InstData->m_vSequences[index]->m_szEnd;
				}
				return NULL;
				break;

			case 3:
				//*** FPS column
				if(index == -1)
				{
					//*** Return the column title
					return AnimLBColumnHeadings[column];

				} else if(index >= 0 && index < data->m_InstData->GetAnimSequenceCount() )
				{
					return data->m_InstData->m_vSequences[index]->m_szFPS;
				}
				return NULL;
				break;

			case 4:
				//*** Cyclic column
				if(index == -1)
				{
					//*** Return the column title
					return AnimLBColumnHeadings[column];

				} else if(index >= 0 && index < data->m_InstData->GetAnimSequenceCount() )
				{
					return AnimLBCyclic[data->m_InstData->m_vSequences[index]->m_nCyclic];
				}
				return NULL;
				break;

			default:
				return NULL;
				break;
			}

		} else
		{
			return NULL;
		}
	} else
	{
		return NULL;
	}
}

//*** Return the width of a given column for the Anim list box
static int AnimListBoxColumnWidth( void *userdata, int column )
{
	if(userdata)
	{
 		CGUIMeshSelector* data = (CGUIMeshSelector*)userdata;

		if(column < AnimMaxColumns)
		{
			//*** If Layout is asking for a column we want defined...
			return AnimLBColumnWidths[column];
		} else
		{
			//*** ...otherwise return 0 to indicate a column we don't need
			return 0;
		}

	} else
	{
		return 0;
	}
}

//*** Event callback for the Anim list box
static void AnimationListBoxEvent( LWControl *ectl, void *edata )
{
	LWPanControlDesc desc;
	LWValue ival    = { LWT_INTEGER }, ivecval = { LWT_VINT }, fval    = { LWT_FLOAT }, fvecval = { LWT_VFLOAT }, sval    = { LWT_STRING };

	CGUIMeshSelector* data = (CGUIMeshSelector*)edata;
	int sel;

	//*** Retrieve the selected row from the list box and store it
	ectl->get( ectl, CTL_VALUE, &ival );
	sel = (int) ival.intv.value;
//	data->m_InstData->m_nLastSelected = sel;
	data->m_nAnimSelectedRow = sel;

	//*** Change the text of the edit controls to match the selection
	SET_STR( data->m_AnimNameEdit, (data->m_InstData->m_vSequences[sel]->m_szName), strlen(data->m_InstData->m_vSequences[sel]->m_szName));
	SET_INT( data->m_AnimFPSEdit, (data->m_InstData->m_vSequences[sel]->m_nFPS) );
	SET_INT( data->m_AnimStartEdit, (data->m_InstData->m_vSequences[sel]->m_nStart) );
	SET_INT( data->m_AnimEndEdit, (data->m_InstData->m_vSequences[sel]->m_nEnd) );
	SET_INT( data->m_AnimCyclicChoice, (data->m_InstData->m_vSequences[sel]->m_nCyclic) );

	//*** Update the buttons
	data->SetAnimationCanvas(DR_RENDER);
}

//*** New Sequence event callback
static void AnimNewButtonEvent( LWControl *ectl, void *edata )
{
	LWPanControlDesc desc;
	LWValue	ival    = { LWT_INTEGER }, ivecval = { LWT_VINT }, fval    = { LWT_FLOAT }, fvecval = { LWT_VFLOAT }, sval    = { LWT_STRING };

	CGUIMeshSelector* data = (CGUIMeshSelector*)edata;

	//*** Build a new animation sequnce and refresh the list
	int num = data->m_InstData->NewAnimSequence();
	data->RefreshAnimList();

	//*** Update the buttons
	data->SetAnimationCanvas(DR_RENDER);
}

//*** Delete Sequence event callback
static void AnimDeleteButtonEvent( LWControl *ectl, void *edata )
{
	LWPanControlDesc desc;
	LWValue	ival    = { LWT_INTEGER }, ivecval = { LWT_VINT }, fval    = { LWT_FLOAT }, fvecval = { LWT_VFLOAT }, sval    = { LWT_STRING };

	CGUIMeshSelector* data = (CGUIMeshSelector*)edata;

	//*** Delete the selected row
	if(data->m_nAnimSelectedRow > -1)
	{
		data->m_InstData->DeleteAnimSequence(data->m_nAnimSelectedRow);
		data->RefreshAnimList();
	}

	//*** Update the buttons
	data->SetAnimationCanvas(DR_RENDER);
}

//*** Move sequence up event callback
static void AnimUpButtonEvent( LWControl *ectl, void *edata )
{
	LWPanControlDesc desc;
	LWValue	ival    = { LWT_INTEGER }, ivecval = { LWT_VINT }, fval    = { LWT_FLOAT }, fvecval = { LWT_VFLOAT }, sval    = { LWT_STRING };

	CGUIMeshSelector* data = (CGUIMeshSelector*)edata;

	//*** Move the sequence up a row
	if(data->m_nAnimSelectedRow > -1)
	{
		data->m_InstData->MoveAnimSequenceUp(data->m_nAnimSelectedRow);
	}

	//*** Update the buttons
	data->SetAnimationCanvas(DR_RENDER);

}

//*** Move sequence down event callback
static void AnimDownButtonEvent( LWControl *ectl, void *edata )
{
	LWPanControlDesc desc;
	LWValue	ival    = { LWT_INTEGER }, ivecval = { LWT_VINT }, fval    = { LWT_FLOAT }, fvecval = { LWT_VFLOAT }, sval    = { LWT_STRING };

	CGUIMeshSelector* data = (CGUIMeshSelector*)edata;

	//*** Move the sequence down a row
	if(data->m_nAnimSelectedRow > -1)
	{
		data->m_InstData->MoveAnimSequenceDown(data->m_nAnimSelectedRow);
	}

	//*** Update the buttons
	data->SetAnimationCanvas(DR_RENDER);
}

//*** Event for the m_AnimNameEdit control
static void AnimNameEditEvent( LWControl *ectl, void *edata )
{
	LWPanControlDesc desc;
	LWValue	ival = { LWT_INTEGER }, ivecval = { LWT_VINT }, fval = { LWT_FLOAT }, fvecval = { LWT_VFLOAT }, sval    = { LWT_STRING };

	char strbuf[MAX_NODE_NAME];
 	CGUIMeshSelector* data = (CGUIMeshSelector*)edata;

	GET_STR( ectl, strbuf, MAX_NODE_NAME);

	//*** Store the new name into the row if a row has been selected
	if(data->m_nAnimSelectedRow > -1)
	{
		strncpy(data->m_InstData->m_vSequences[data->m_nAnimSelectedRow]->m_szName,strbuf,MAX_NODE_NAME);
		data->RefreshAnimList();
	}

}

//*** Event for the m_AnimFPSEdit control
static void AnimFPSEditEvent( LWControl *ectl, void *edata )
{
	LWPanControlDesc desc;
	LWValue ival = { LWT_INTEGER },ivecval = { LWT_VINT },fval = { LWT_FLOAT },fvecval = { LWT_VFLOAT },sval    = { LWT_STRING };

	CGUIMeshSelector* data = (CGUIMeshSelector*)edata;
	int value;

	GET_INT(ectl, value);

	//*** Store the new FPS into the row if a row has been selected
	if(data->m_nAnimSelectedRow > -1)
	{
		data->m_InstData->m_vSequences[data->m_nAnimSelectedRow]->m_nFPS = value;
		char strbuf[MAX_NODE_NAME];
		sprintf(strbuf,"%i",value);
		strncpy(data->m_InstData->m_vSequences[data->m_nAnimSelectedRow]->m_szFPS,strbuf,64);
		data->RefreshAnimList();
	}
}

//*** Event for the m_AnimStartEdit control
static void AnimStartEditEvent( LWControl *ectl, void *edata )
{
	LWPanControlDesc desc;
	LWValue ival = { LWT_INTEGER },ivecval = { LWT_VINT },fval = { LWT_FLOAT },fvecval = { LWT_VFLOAT },sval    = { LWT_STRING };

	CGUIMeshSelector* data = (CGUIMeshSelector*)edata;
	int value;

	GET_INT(ectl, value);

	//*** Store the new Start into the row if a row has been selected
	if(data->m_nAnimSelectedRow > -1)
	{
		data->m_InstData->m_vSequences[data->m_nAnimSelectedRow]->m_nStart = value;
		char strbuf[MAX_NODE_NAME];
		sprintf(strbuf,"%i",value);
		strncpy(data->m_InstData->m_vSequences[data->m_nAnimSelectedRow]->m_szStart,strbuf,64);
		data->RefreshAnimList();
	}
}

//*** Event for the m_AnimEndEdit control
static void AnimEndEditEvent( LWControl *ectl, void *edata )
{
	LWPanControlDesc desc;
	LWValue ival = { LWT_INTEGER },ivecval = { LWT_VINT },fval = { LWT_FLOAT },fvecval = { LWT_VFLOAT },sval    = { LWT_STRING };

	CGUIMeshSelector* data = (CGUIMeshSelector*)edata;
	int value;

	GET_INT(ectl, value);

	//*** Store the new End into the row if a row has been selected
	if(data->m_nAnimSelectedRow > -1)
	{
		data->m_InstData->m_vSequences[data->m_nAnimSelectedRow]->m_nEnd = value;
		char strbuf[MAX_NODE_NAME];
		sprintf(strbuf,"%i",value);
		strncpy(data->m_InstData->m_vSequences[data->m_nAnimSelectedRow]->m_szEnd,strbuf,64);
		data->RefreshAnimList();
	}
}

//*** Event for the m_AnimCyclicChoice control
static void AnimCyclicChoiceEvent( LWControl *ectl, void *edata )
{
	LWPanControlDesc desc;
	LWValue ival = { LWT_INTEGER },ivecval = { LWT_VINT },fval = { LWT_FLOAT },fvecval = { LWT_VFLOAT },sval    = { LWT_STRING };

	CGUIMeshSelector* data = (CGUIMeshSelector*)edata;
	int value;

	GET_INT(ectl, value);

	//*** Store the new Cyclic into the row if a row has been selected
	if(data->m_nAnimSelectedRow > -1)
	{
		data->m_InstData->m_vSequences[data->m_nAnimSelectedRow]->m_nCyclic = value;
		data->RefreshAnimList();
	}
}


//************************************************************
//*** Functions for the Advanced Tab
//************************************************************

//*** Event for PolyPointTypeChoice control
static void PolyPointTypeChoiceEvent( LWControl *ectl, void *edata )
{
	LWPanControlDesc desc;
	LWValue ival    = { LWT_INTEGER },ivecval = { LWT_VINT },fval    = { LWT_FLOAT },fvecval = { LWT_VFLOAT },sval    = { LWT_STRING };

	CGUIMeshSelector* data = (CGUIMeshSelector*)edata;
	int value;

	GET_INT(ectl, value);
	data->m_InstData->m_nPolyPointType = value;
}

//*** Event for the BaseTransFrameEdit control
static void BaseTransFrameEditEvent( LWControl *ectl, void *edata )
{
	LWPanControlDesc desc;
	LWValue ival    = { LWT_INTEGER },ivecval = { LWT_VINT },fval    = { LWT_FLOAT },fvecval = { LWT_VFLOAT },sval    = { LWT_STRING };

	CGUIMeshSelector* data = (CGUIMeshSelector*)edata;
	float value;

	GET_FLOAT(ectl, value);
	data->m_InstData->m_fBaseTransFrame = value;
}

//*** Event for ExportMeshAsNodeChoice control
static void ExportMeshAsNodeChoiceEvent( LWControl *ectl, void *edata )
{
	LWPanControlDesc desc;
	LWValue ival    = { LWT_INTEGER },ivecval = { LWT_VINT },fval    = { LWT_FLOAT },fvecval = { LWT_VFLOAT },sval    = { LWT_STRING };

	CGUIMeshSelector* data = (CGUIMeshSelector*)edata;
	int value;

	GET_INT(ectl, value);
	data->m_InstData->m_nExportMeshAsNode = value;
}

//*** Event for the MeshNodeExtEdit control
static void MeshNodeExtEditEvent( LWControl *ectl, void *edata )
{
	LWPanControlDesc desc;
	LWValue ival    = { LWT_INTEGER },ivecval = { LWT_VINT },fval    = { LWT_FLOAT },fvecval = { LWT_VFLOAT },sval    = { LWT_STRING };

	char strbuf[MAX_OBJECT_NAME];
 	CGUIMeshSelector* data = (CGUIMeshSelector*)edata;

	GET_STR( ectl, strbuf, MAX_OBJECT_NAME);

	strncpy(data->m_InstData->m_szMeshNodeExt,strbuf,MAX_OBJECT_NAME);

}


//************************************************************
//*** Panel callbacks
//************************************************************

//*** Panel draw callback
static void PanelDrawCallback( LWPanelID panel, void *edata, DrMode mode )
{
	LWPanControlDesc desc;
	LWValue
		ival    = { LWT_INTEGER },
		ivecval = { LWT_VINT },
		fval    = { LWT_FLOAT },
		fvecval = { LWT_VFLOAT },
		sval    = { LWT_STRING };

	CGUIMeshSelector* data = (CGUIMeshSelector*)edata;

	int pos;
	int width = PAN_GETW( data->m_Panf, panel );

	pos = CON_Y( data->m_GlobalTabChoice ) - 12;
	data->m_Panf->drawFuncs->drawBorder(panel, COLOR_BLACK, 0, pos, width, 0);

	pos = CON_Y( data->m_GlobalTabChoice ) - 8;
	data->m_Panf->drawFuncs->drawBorder(panel, COLOR_BLACK, 0, pos, width, 0);

	pos = CON_Y( data->m_GlobalTabChoice );
	pos += CON_H( data->m_GlobalTabChoice ) - 6;
	data->m_Panf->drawFuncs->drawBorder(panel, COLOR_BLACK, 0, pos, width, 0);

	pos = CON_Y( data->m_GlobalTabChoice );
	pos += CON_H( data->m_GlobalTabChoice ) + 244;
	data->m_Panf->drawFuncs->drawBorder(panel, COLOR_BLACK, 0, pos, width, 0);

	pos = CON_Y( data->m_GlobalTabChoice );
	pos += CON_H( data->m_GlobalTabChoice ) + 248;
	data->m_Panf->drawFuncs->drawBorder(panel, COLOR_BLACK, 0, pos, width, 0);
}

//*** Panel user open callback
static void PanelUserOpenCallback( LWPanelID panel, void *edata )
{
	LWPanControlDesc desc;
	LWValue ival    = { LWT_INTEGER }, ivecval = { LWT_VINT }, fval    = { LWT_FLOAT }, fvecval = { LWT_VFLOAT }, sval    = { LWT_STRING };

	CGUIMeshSelector* data = (CGUIMeshSelector*)edata;

	//SET_INT( data->m_MeshListBox, data->m_InstData->m_nLastSelected );
}

//*** Panel user close callback
static void PanelUserCloseCallback( LWPanelID panel, void *edata )
{
	LWPanControlDesc desc;
	LWValue ival    = { LWT_INTEGER }, ivecval = { LWT_VINT }, fval    = { LWT_FLOAT }, fvecval = { LWT_VFLOAT }, sval    = { LWT_STRING };

	CGUIMeshSelector* data = (CGUIMeshSelector*)edata;

	GET_INT( data->m_MeshListBox, data->m_InstData->m_nLastSelected );
}