/*	LWANIMCUSTOMOBJ.CPP
**
**	Functions for the anim sequence custom object plug-in used by the DTS exporter plug-in.
**	This plug-in can be used on a Null object in LW and will display the animation
**	sequence name that is defined in the Master Handler DTS exporter plug-in for the
**	current frame.
**
**	Revision History:
**	April 9, 2002		David Wyand		Created file
**	April 26, 2002		David Wyand		Remove 'static' function declarations based on the
**										Mac Code Warrior build by C J Silverio.
**	April 26, 2002		David Wyand		Added second line of text that displays the current
**										animation sequence's start and end frames.
*/

//*** Includes
#include <stdio.h>
#include <lwserver.h>			// all plug-ins need this
#include <lwhandler.h>
#include <lwrender.h>
#include <lwcustobj.h>
#include <lwpanel.h>			// for "classic" panels
#include <lwhost.h>				// for the LWMessageFuncs global
#include "LWGlobal.h"
#include "LWAnimCustomObj.h"
#include "LWACOData.h"
#include "GUIAnimCustomObj.h"
#include "LWData.h"

//*** Global variables
GlobalFunc *g_animCustObjGlobal;			// Pointer to all global functions
LWPanelFuncs *animCustObjPanf;				// the panel functions
LWPanelID animCustObjPanel = 0;				// the panel

//*** Prototypes
extern "C"
{
XCALL_(const char*)		ACODescribe(LWInstance);
XCALL_(LWInstance)		ACOCreate(void *,void *,LWError *);
XCALL_(void)			ACODestroy(LWInstance);
XCALL_(LWError)			ACOCopy(LWInstance,LWInstance);
XCALL_(LWError)			ACOLoad(LWInstance,const LWLoadState *);
XCALL_(LWError)			ACOSave(LWInstance,const LWSaveState *);
XCALL_(LWError)			ACOInit(LWInstance, int);
XCALL_(void)			ACOCleanUp(LWInstance);
XCALL_(LWError)			ACONewTime(LWInstance, LWFrame, LWTime);
XCALL_(void)			ACOEvaluate(LWInstance,const LWCustomObjAccess *);
XCALL_(unsigned int)	ACOFlags(LWInstance);
XCALL_(LWError)			ACOOptions(LWInstance inst);
}


//************************************************************
//*** ACTIVATE()
//*** Layout calls this when the user selects the plugin from the list of
//*** custom object plug-ins.
//************************************************************
extern "C"
{
//************
XCALL_(int) AnimCustObjActivate( long version, GlobalFunc *global, LWCustomObjHandler *local, void *serverData )
{
	//*** Check that Layout is of the correct version
	if ( version != LWCUSTOMOBJ_VERSION )
		return AFUNC_BADVERSION;

	//*** Pass along pointers to the different plug-in functions (generic)
	if(local->inst)
	{
		local->inst->create     = ACOCreate;
		local->inst->destroy    = ACODestroy;
		local->inst->load       = ACOLoad;
		local->inst->save       = ACOSave;
		local->inst->copy       = ACOCopy;
		local->inst->descln     = ACODescribe;
	}

	//*** Pass along pointers to the different plug-in render functions (shader)
	if(local->rend)
	{
		local->rend->init     = ACOInit;
		local->rend->cleanup  = ACOCleanUp;
		local->rend->newTime  = ACONewTime;
	}
	local->evaluate         = ACOEvaluate;
	local->flags            = ACOFlags;

	//*** Get the panels functions
	animCustObjPanf = (LWPanelFuncs *) (*global)( LWPANELFUNCS_GLOBAL, GFUSE_TRANSIENT );
	if ( !animCustObjPanf )
		return AFUNC_BADGLOBAL;

	//*** Initalize the CLWGlobal static class
	if( CLWGlobal::Initialize(global) == false)
	{
		return AFUNC_BADGLOBAL;
	}

	g_animCustObjGlobal = global;

	return AFUNC_OK;
}

//************************************************************
//*** ACOCREATE()
//*** Called when an instance of the plug-in has been created
//************************************************************
XCALL_(LWInstance) ACOCreate(void *priv, void *context, LWError *err)
{
	CLWACOData*	mydata;

	mydata = new CLWACOData;

	return(mydata);
}

//************************************************************
//*** ACODESTROY()
//*** Called when an instance of the plug-in has been destroyed
//************************************************************
XCALL_(void) ACODestroy(LWInstance inst)
{
	CLWACOData*	mydata;

	mydata = (CLWACOData*) inst;

	if(mydata)
	{
		delete mydata;
	}

}

//************************************************************
//*** ACOCOPY()
//*** Called when an instance of the plug-in needs to be copied
//************************************************************
XCALL_(LWError) ACOCopy(LWInstance d1,LWInstance d2)
{
	CLWACOData*	source;
	CLWACOData*	dest;

	source = (CLWACOData*) d2;
	dest = (CLWACOData*) d1;

	dest->m_fTextPos = source->m_fTextPos;
	dest->m_nTextJustification = source->m_nTextJustification;

    return((LWError) NULL);
}

//************************************************************
//*** ACOLOAD()
//*** Called to load in the plug-in data from a scene
//************************************************************
XCALL_(LWError) ACOLoad(LWInstance inst,const LWLoadState *lState)
{
	CLWACOData*	mydata;

	mydata = (CLWACOData*) inst;

	mydata->Load(lState);

    return((LWError) NULL);
}

//************************************************************
//*** ACOSAVE()
//*** Called to save the plug-in data to a scene
//************************************************************
XCALL_(LWError) ACOSave(LWInstance inst,const LWSaveState *sState)
{
	CLWACOData*	mydata;

	mydata = (CLWACOData*) inst;

	mydata->Save(sState);

    return((LWError) NULL);
}

//************************************************************
//*** ACODESCRIBE()
//*** Called when Layout needs to display the name of the plug-in to the user
//************************************************************
XCALL_(const char *) ACODescribe(LWInstance inst)
{
    return("DTS Exporter Anim Sequence Cust Obj");
}

//************************************************************
//*** ACOINIT()
//*** Called prior to rendering to perform any setup
//************************************************************
XCALL_( LWError ) ACOInit( LWInstance inst, int mode )
{

	return (LWError) NULL;
}

//************************************************************
//*** ACOCLEANUP()
//*** Called after rendering to perform any cleanup
//************************************************************
XCALL_( void ) ACOCleanUp( LWInstance inst )
{
	return;
}

//************************************************************
//*** ACONEWTIME()
//*** Called at the start of each sampling pass
//************************************************************
XCALL_( LWError ) ACONewTime( LWInstance inst, LWFrame f, LWTime t )
{
	return (LWError) NULL;
}

//************************************************************
//*** ACOFLAGS()
//*** Called to tell layout what types of events we should be sent.
//************************************************************
XCALL_(unsigned int) ACOFlags(LWInstance inst)
{
	return(0);
}

//************************************************************
//*** ACOEVALUATE()
//*** Where the actual rendering happens.
//************************************************************
XCALL_( void ) ACOEvaluate( LWInstance inst, const LWCustomObjAccess *coa )
{
	char text[1024];
	CLWACOData*	mydata;
	double pos[3];
	CLWData* masterdata = CLWData::FindFirstInstance();

	mydata = (CLWACOData*) inst;
	pos[0]=0.0;
	pos[1]=mydata->m_fTextPos;
	pos[2]=0.0;

	//*** Obtain the animation sequence name for the current frame
	LWFrame curframe= CLWGlobal::timeinfo->frame;
	if(masterdata != NULL)
	{
		coa->text(coa->dispData,pos,masterdata->GetFirstAnimSequenceName(curframe),mydata->m_nTextJustification,LWCSYS_OBJECT);

		sprintf(text,"Start: %s, End: %s",masterdata->GetFirstAnimSequenceStart(curframe),masterdata->GetFirstAnimSequenceEnd(curframe));
		if(coa->view == LWVIEW_XZ)
		{
			//*** Adjust second line position for top and bottom views
			//*** TODO: Modify so that the text stays a constant distance from preceding
			//*** test no matter what the view's zoom level is.  This hardcoded value
			//*** only works for a limited zoom range.
			pos[2] = pos[2] - 0.1;
		} else
		{
			//*** Adjust second line position for all other views
			//*** TODO: Modify so that the text stays a constant distance from preceding
			//*** test no matter what the view's zoom level is.  This hardcoded value
			//*** only works for a limited zoom range.
			pos[1] = pos[1] - 0.1;
		}
		coa->text(coa->dispData,pos,text,mydata->m_nTextJustification,LWCSYS_OBJECT);

	} else
	{
		strcpy(text,"DTS Exporter Not Found");
		coa->text(coa->dispData,pos,text,mydata->m_nTextJustification,LWCSYS_OBJECT);
	}
}

} // extern "C"

//************************************************************
//*** INTERFACE()
//*** Layout calls this when it needs info about the GUI
//************************************************************
extern "C"
{
XCALL_(int) AnimCustObjInterface(long version, GlobalFunc *global, void *local, void *serverData )
{
	LWInterface* mylocal;
	
	if ( version != LWINTERFACE_VERSION )
		return AFUNC_BADVERSION;

	mylocal = (LWInterface*) local;

	mylocal->panel   = NULL;
	mylocal->options = ACOOptions;
	mylocal->command = 0; // (NULL);

	if(animCustObjPanf)
	{
		animCustObjPanf->globalFun = global;
	}

	return(AFUNC_OK);
}

//************************************************************
//*** ACOOPTIONS()
//*** Layout calls this when the user opens the GUI
//************************************************************
XCALL_(LWError) ACOOptions(LWInstance inst)
{
	CGUIAnimCustomObj custobjgui;
	CLWACOData* instData = (CLWACOData*) inst;

	//*** Define the panel
	if( !( animCustObjPanel = PAN_CREATE( animCustObjPanf, CLWGlobal::GetPanelTitle() )))
	{
		CLWGlobal::msg->info("Cannot open the user interface!","");
		return 0;
	}

	//*** Open the panel
	if(animCustObjPanel)
	{
		//*** Create all of the controls
		custobjgui.CreateAnimCustomObj(instData, animCustObjPanf, animCustObjPanel, 0, 0, 450, 200);

		//*** Display the panel
		animCustObjPanf->open( animCustObjPanel, PANF_BLOCKING );

		PAN_KILL( animCustObjPanf, animCustObjPanel );
	}

	return(AFUNC_OK);
}

} // extern "C"
