/*	LWDATA.CPP
**
**	Supports the CLWData class
**
**	Revision History:
**	February 11, 2002	David Wyand		Created file
**	March 12, 2002		David Wyand		Added code to support the building of a global node list.
**	March 20, 2002		David Wyand		Added support for saving and loading: m_fBaseTransFrame,
**										m_nExportMeshAsNode, and m_szMeshNodeExt.  This has
**										changed the plug-in save version to 2.
**	April 4, 2002		David Wyand		Changed the node functions to support the new members
**										for weight maps for bones.  This changed the root node
**										creation in BuildNodeList() and the bone node creation
**										in AddNodeToList().
**	April 4, 2002		David Wyand		Add code to only load or save the object list if there
**										are more than zero objects in the list.  DOH!  I sure
**										hope this won't break anything for someone.
**	April 4, 2002		David Wyand		Fix a bug in AddNodeToList() that was generating an
**										incorrect parent for a node.
**	April 9, 2002		David Wyand		Added support for a global list of CLWData classes
*/

//*** Includes
#include <cassert>
#include "math.h"
#include "LWGlobal.h"
#include "LWData.h"

//************************************************************
//*** Constructor
//************************************************************
CLWData::CLWData()
{
	m_nPolyPointType = 0;	// Set the point and poly type to original locations
	m_fBaseTransFrame = 0.0f;	// Set the based node tranformation frame to frame 0
	m_nExportMeshAsNode = 1;	// Default to export meshes as nodes
	strncpy(m_szMeshNodeExt,"-node",MAX_OBJECT_NAME);	// Default mesh-node extension

	m_nCollisionType = 0;					// Collision mesh type: 0=None, 1=Bounding Box, 2=Bounding Cylinder, 3=Mesh
	m_nCollisionVisible = 0;				// Display collision meshes
	m_fCollisionComplexity = 0.2f;
	m_nExportMaterials = 1;					// Include material in the file
	strncpy(m_szFilename,"",MAX_FILE_PATH);	// DTS file name from the user
	m_fScaleFactor = 1.0f;
	m_nMinPixelSize = 0;

	m_nExportAnimSequences = 0;	// Default to not export anim sequences

	m_nLastSelected = 0;

	m_nRootNode = -1;

	m_vList.push_back(this);	// Put this CLWData class onto the global class list

}

//************************************************************
//*** Destructor
//************************************************************
CLWData::~CLWData()
{
	vector<CLWData*>::iterator CLWDataItor;
	vector<ObjectInfo*>::iterator itor;

	//*** Clear out the node list
	ClearNodeList();

	//*** Clear the animation sequences
	ClearAnimSequence();

	//*** Delete the MeshSelectionNode instances.  Only need to do this for the m_ItemList
	//*** as m_Nodes points to the same instances.
	for(itor = m_vObjectList.begin(); itor != m_vObjectList.end(); ++itor)
	{
		delete (*itor);
	}

	m_vObjectList.clear();

	//*** Remove this from the list
	for(CLWDataItor = m_vList.begin(); CLWDataItor != m_vList.end(); ++CLWDataItor)
	{
		if((*CLWDataItor) == this)
		{
			(*CLWDataItor) = (CLWData*)NULL;
		}
	}

}

//************************************************************
//*** REFRESHOBJECTLIST()
//*** Refreshes the object list if there are new objects to add
//************************************************************
void CLWData::RefreshObjectList()
{
	LWItemID	id;
	vector<ObjectInfo*>::iterator itor;

	//*** Add any new objects
	id = CLWGlobal::iteminfo->first( LWI_OBJECT, NULL );
	while(id != LWITEM_NULL)
	{
		bool check = false;
		for(itor = m_vObjectList.begin(); itor != m_vObjectList.end(); ++itor)
		{
			if((*itor)->m_ID == id)
			{
				check = true;
				break;
			}
		}

		//*** We did not find our object in the list, so add it to the end
		if(check == false)
		{
			ObjectInfo* node = BuildNode(id);
			m_vObjectList.push_back(node);
		}

		id = CLWGlobal::iteminfo->next(id);
	}

}

//************************************************************
//*** BUILDNODE()
//*** Builds out a single node to add to the object list
//************************************************************
CLWData::ObjectInfo* CLWData::BuildNode(LWItemID id)
{
	ObjectInfo* node = new ObjectInfo;
	node->m_ID = id;
	strncpy(node->m_szLWName,CLWGlobal::iteminfo->name(id), MAX_OBJECT_NAME);
	strcpy(node->m_szDTSName,"");
	node->m_nExportType = 0;
	node->m_nGlobalRotation = 0;
	node->m_nEncodedNormals = 0;

	return (node);
}

//************************************************************
//*** DELETEOBJECT()
//*** Delete an object from the list
//************************************************************
void CLWData::DeleteObject(LWItemID id)
{
	vector<ObjectInfo*> temp;	// For storing the new compact list
	vector<ObjectInfo*>::iterator itor;

	for(itor = m_vObjectList.begin(); itor != m_vObjectList.end();)
	{
		if((*itor)->m_ID == id)
		{
			//*** Delete this object
			delete (*itor);
			itor = m_vObjectList.erase(itor);
		} else
		{
			temp.push_back(*itor);
			++itor;
		}
	}

	//*** Now swap the temp vector with the original one, thus making a new compact
	//*** object list
	m_vObjectList.swap(temp);

}

//************************************************************
//*** CHANGEOBJECTID()
//*** Change an object's ID
//************************************************************
void CLWData::ChangeObjectID(LWItemID oldid, LWItemID newid)
{
	vector<ObjectInfo*>::iterator itor;

	for(itor = m_vObjectList.begin(); itor != m_vObjectList.end();++itor)
	{
		if((*itor)->m_ID == oldid)
		{
			//*** Change from the old ID to the new ID
			(*itor)->m_ID = newid;
			break;
		}
	}
}

//************************************************************
//*** REFRESHOBJECTNAMES()
//*** Refreshes the names of the objects in the list
//************************************************************
void CLWData::RefreshObjectNames()
{
	vector<ObjectInfo*>::iterator itor;

	for(itor = m_vObjectList.begin(); itor != m_vObjectList.end(); ++itor)
	{
		strncpy((*itor)->m_szLWName,CLWGlobal::iteminfo->name((*itor)->m_ID), MAX_OBJECT_NAME);
	}

}

//************************************************************
//*** GETOBJECTCOUNT()
//*** Returns the number of objects
//************************************************************
int CLWData::GetObjectCount()
{
	int count = 0;
	vector<ObjectInfo*>::iterator itor;

	for(itor = m_vObjectList.begin(); itor != m_vObjectList.end(); ++itor)
	{
		++count;
	}

	return count;
}

//************************************************************
//*** GETOBJECTEXPORTTYPE()
//*** Returns the export type for an object
//************************************************************
int CLWData::GetObjectExportType(LWItemID id)
{
	vector<ObjectInfo*>::iterator itor;

	for(itor = m_vObjectList.begin(); itor != m_vObjectList.end(); ++itor)
	{
		if( (*itor)->m_ID == id)
		{
			return (*itor)->m_nExportType;
		}
	}

	return ET_None;
}

//************************************************************
//*** ISINOBJECTLIST()
//*** Returns true if the object is in the object list
//************************************************************
bool CLWData::IsInObjectList(LWItemID id)
{
	vector<ObjectInfo*>::iterator itor;

	for(itor = m_vObjectList.begin(); itor != m_vObjectList.end(); ++itor)
	{
		if( (*itor)->m_ID == id)
		{
			return (true);
		}
	}

	return (false);
}

//************************************************************
//*** GETOBJECTBYNAME()
//*** Returns the LW ID of the object given its name, or 0 if none
//************************************************************
LWItemID CLWData::GetObjectByName(char* name)
{
	LWItemID	id;

	id = CLWGlobal::iteminfo->first( LWI_OBJECT, NULL );
	while(id != LWITEM_NULL)
	{
		if( !strcmp(CLWGlobal::iteminfo->name(id),name) )
		{
			return (id);
		}

		id = CLWGlobal::iteminfo->next(id);
	}

	return 0;
}

//************************************************************
//*** GETOBJECTUSERNAME()
//*** Returns the objects DTS or LW name as appropriate
//************************************************************
char* CLWData::GetObjectUserName(LWItemID id)
{
	vector<ObjectInfo*>::iterator itor;

	for(itor = m_vObjectList.begin(); itor != m_vObjectList.end(); ++itor)
	{
		if( (*itor)->m_ID == id)
		{
			if(strlen((*itor)->m_szDTSName) == 0)
			{
				return (*itor)->m_szLWName;
			} else
			{
				return (*itor)->m_szDTSName;
			}
		}
	}

	return (char*)NULL;
}

//************************************************************
//*** GETFIRSTCOLLISIONOBJECT()
//*** Returns the LW ID of the object, or 0 if none.
//************************************************************
LWItemID CLWData::GetFirstCollisionObject()
{
	vector<ObjectInfo*>::iterator itor;

	for(itor = m_vObjectList.begin(); itor != m_vObjectList.end(); ++itor)
	{
		if( (*itor)->m_nExportType == ET_Collision)
		{
			return (*itor)->m_ID;
		}
	}

	return 0;
}

//************************************************************
//*** ADDSURFACE()
//*** Add a LW surface ID to the global list
//************************************************************
void CLWData::AddSurface(LWSurfaceID surf)
{
	m_vSurfaces.push_back(surf);
}

//************************************************************
//*** GETSURFACEINDEX()
//*** Retrieve the index in the global list of a LW surface ID
//************************************************************
int CLWData::GetSurfaceIndex(LWSurfaceID surf)
{
	vector<LWSurfaceID>::iterator itor;
	int count = -1;

	for(itor = m_vSurfaces.begin(); itor != m_vSurfaces.end(); ++itor)
	{
		++count;
		if((*itor) == surf)
		{
			return count;
		}
	}

	return -1;
}

//************************************************************
//*** CLEARSURFACES()
//*** Remove all surfaces from the global list
//************************************************************
void CLWData::ClearSurfaces()
{
	m_vSurfaces.clear();
}

//************************************************************
//*** BUILDNODELIST()
//*** Build a list of all nodes in the scene.  These may be NULLs or bones.
//************************************************************
void CLWData::BuildNodeList()
{
	int i;
	int objnum = 0;
	LWItemID curobj;

	//*** Check if a Root node is required, and build it if it is
	if(DefineRootNode() == true)
	{
		char buffer[10];

		NodeInfo* node = new NodeInfo;
		node->m_ID = NULL;
		strncpy(node->m_szName,"Root",MAX_OBJECT_NAME);
		node->m_Parent = LWITEM_NULL;
		node->m_nParentNode = -1;
		strncpy(node->m_szParent,"-",MAX_OBJECT_NAME);
		sprintf(buffer,"%i",objnum);
		strncpy(node->m_szNumber,buffer,10);
		node->m_nNumber = objnum;
		node->m_nType = NT_NULL;
		node->m_vPosition[0] = 0.0;
		node->m_vPosition[1] = 0.0;
		node->m_vPosition[2] = 0.0;
		node->m_vRotation[0] = 0.0;
		node->m_vRotation[1] = 0.0;
		node->m_vRotation[2] = 0.0;
		node->m_ParentObject = NULL;
		strncpy(node->m_szWeightMap,"",MAX_OBJECT_NAME);

		m_vNodes.push_back(node);
		m_nRootNode = objnum;
		++objnum;

	} else
	{
		m_nRootNode = -1;
	}

	//*** Go through all of the objects
	int numMeshes = GetObjectCount();
	for (i = 0 ; i < numMeshes ; i++)
	{
		bool meshexport = false;
		LWItemID objID = m_vObjectList[i]->m_ID;

		//*** If this object is to not export by the user, then don't process it or it's children
		if(m_vObjectList[i]->m_nExportType == CLWData::ET_None)
		{
			continue;
		}

		//*** Check if we've already been added to the list as a safety check
		if(DoesNodeExist(objID))
		{
			//*** We're already in the list, so move on
			continue;
		}

		//*** Do we have a parent?  If so, we are not added here
		if(CLWGlobal::iteminfo->parent(objID) != LWITEM_NULL)
		{
			continue;
		}

		//*** Check if this object should be exported and perform the export
		meshexport = ConditionalAddObjectToNodeList(objnum, objID, true);

		//*** Add any bones
		curobj = CLWGlobal::iteminfo->first(LWI_BONE, objID);
		while(curobj != LWITEM_NULL)
		{
			//*** Make sure the bone doesn't yet exist and is a direct child of the mesh
			if(!DoesNodeExist(curobj) && CLWGlobal::iteminfo->parent(curobj) == objID)
			{
				AddNodeToList(objnum, curobj, false, meshexport);
				++objnum;
			}

			//*** Build out any children of this bone
			BuildNodeChildList(objnum, curobj, true);

			//*** Go onto the next sibling
			LWItemID oldobj = curobj;
			curobj = CLWGlobal::iteminfo->next(curobj);
		}

		//*** Add any children
		BuildNodeChildList(objnum, objID, meshexport);
	}
}

//************************************************************
//*** CLEARNODELIST()
//*** Clear the list of all nodes in the scene.
//************************************************************
void CLWData::ClearNodeList()
{
	vector<NodeInfo*>::iterator itor;

	for(itor = m_vNodes.begin(); itor != m_vNodes.end(); ++itor)
	{
		delete (*itor);
	}

	m_vNodes.clear();

	m_nRootNode = -1;

}

//************************************************************
//*** BUILDNODECHILDLIST()
//*** Builds out the children node list of a given object
//************************************************************
void CLWData::BuildNodeChildList(int &objnum, LWItemID objID, bool localcoords)
{
	LWItemID curobj;
	LWItemID boneobj;
	bool meshexport = false;

	curobj = CLWGlobal::iteminfo->firstChild(objID);

	while(curobj != LWITEM_NULL)
	{

		//*** If this is an object and is set to not export by the user, then don't process it or it's children
		if(IsInObjectList(curobj) && GetObjectExportType(curobj) == CLWData::ET_None)
		{
			goto NextSibling;
		}

		//*** Make sure the child doesn't yet exist and is a direct child of the mesh
		if(!DoesNodeExist(curobj) && CLWGlobal::iteminfo->parent(curobj) == objID)
		{
			if( CLWGlobal::iteminfo->type(curobj) == LWI_BONE )
			{
				AddNodeToList(objnum, curobj, false, localcoords);
				++objnum;
				meshexport = true;

			} else
			{
				//*** Export the object if it should be...
				meshexport = ConditionalAddObjectToNodeList(objnum, curobj, localcoords);

				//*** Add any bones
				boneobj = CLWGlobal::iteminfo->first(LWI_BONE, curobj);
				while(boneobj != LWITEM_NULL)
				{
					//*** Make sure the bone doesn't yet exist and is a direct child of the mesh
					if(!DoesNodeExist(boneobj) && CLWGlobal::iteminfo->parent(boneobj) == curobj)
					{
						AddNodeToList(objnum, boneobj, false, meshexport);
						++objnum;
					}

					//*** Build out any children of this bone
					BuildNodeChildList(objnum, boneobj, true);

					//*** Go onto the next sibling
					LWItemID oldobj = boneobj;
					boneobj = CLWGlobal::iteminfo->next(boneobj);
				}

			}
		}

		//*** Build out any children of this object
		BuildNodeChildList(objnum, curobj, meshexport);

		//*** Go onto the next sibling
		NextSibling:
		curobj = CLWGlobal::iteminfo->nextChild(objID, curobj);
	}
}

//************************************************************
//*** CONDITIONALADDOBJECTTONODELIST()
//*** Based on certain conditions, an object will be added to the node list.
//*** Returns true if the object was added
//************************************************************
bool CLWData::ConditionalAddObjectToNodeList(int &objnum, LWItemID objID, bool localcoords)
{
	int exportType = GetObjectExportType(objID);

	if( (m_nExportMeshAsNode == 1 && (exportType == CLWData::ET_Mesh || exportType == CLWData::ET_Collision))	// This is a mesh and the user has chosen to export meshes as nodes
		|| (CLWGlobal::objinfo->numPolygons(objID) == 0 && exportType == CLWData::ET_Mesh)	// This is a LW Null object and has been selected as a mesh object
		|| exportType == CLWData::ET_Node)	// This object has been selected as a node
	{
		//*** Add this object to the list
		AddNodeToList(objnum, objID, true, localcoords);
		++objnum;
		return true;
	}

	return false;
}

//************************************************************
//*** ADDNODETOLIST()
//*** Builds a node and adds it to the node list
//************************************************************
void CLWData::AddNodeToList(int objnum, LWItemID objID, bool isobject, bool localcoords)
{
	char buffer[10];

	NodeInfo* node = new NodeInfo;
	node->m_ID = objID;

	if(isobject)
	{
		strncpy(node->m_szName,GetObjectUserName(objID),MAX_OBJECT_NAME);

		//*** Add an extension if this is a mesh
		if(CLWGlobal::objinfo->numPolygons(objID) != 0)
		{
			strncat(node->m_szName,m_szMeshNodeExt,(MAX_OBJECT_NAME - strlen(node->m_szName)));
		}
	} else
	{
		strncpy(node->m_szName,CLWGlobal::iteminfo->name(objID),MAX_OBJECT_NAME);
	}

	node->m_Parent = CLWGlobal::iteminfo->parent(objID);
	if(node->m_Parent != LWITEM_NULL)
	{
		//*** Check if the parent has been added to the node list.  If not, then the
		//*** parent of this object is the world
		if(!DoesNodeExist(node->m_Parent))
		{
			strncpy(node->m_szParent,"*World*",MAX_OBJECT_NAME);
			node->m_nParentNode = -1;
		} else
		{
			strncpy(node->m_szParent,CLWGlobal::iteminfo->name(node->m_Parent),MAX_OBJECT_NAME);
			node->m_nParentNode = GetNodeNumber(node->m_Parent);
		}
	} else
	{
		strncpy(node->m_szParent,"-",MAX_OBJECT_NAME);
		node->m_nParentNode = -1;
	}

	//*** Get initial position and rotation -- TODO: take bool localcoords into account!!!!!
	if(localcoords)
	{
		LWDVector	Pos, Rot;
		CLWGlobal::iteminfo->param(objID, LWIP_POSITION, m_fBaseTransFrame, Pos);
		CLWGlobal::iteminfo->param(objID, LWIP_ROTATION, m_fBaseTransFrame, Rot);

		node->m_vPosition[0] = (float) Pos[0];
		node->m_vPosition[1] = (float) Pos[1];
		node->m_vPosition[2] = (float) Pos[2];

		node->m_vRotation[0] = (float) Rot[0];
		node->m_vRotation[1] = (float) Rot[1];
		node->m_vRotation[2] = (float) Rot[2];

	} else
	{
		//*** TODO: Handle world coordinates of an object.  Currently only the local
		//*** coordinates are used so any parent that is not exported as a node will
		//*** need to be at the origin.  Otherwise, there will be a discontinuity between
		//*** the DTS file and LightWave

		/*
		You can get a set of angles from the up and forward vectors.

		p = -asin( forward[ 1 ])
		h = asin( forward[ 0 ] / cos( p ))
		b = acos( up[ 1 ] / cos( p ))

		p = 90 degrees is obviously a special case.

		But keep in mind that what you're getting is only an orientation. You
		can't use it by itself to bake the item's rotation, because each parent
		can rotate around a different pivot point. And it won't be able to
		represent multiples of the principle interval--headings outside the 0
		to 360 range, for example.
		*/
		/*
		LWDVector vRight,vUp,vForward;

		CLWGlobal::iteminfo->param(objID, LWIP_W_POSITION, m_fBaseTransFrame, node->m_vPosition);
		CLWGlobal::iteminfo->param(objID, LWIP_W_RIGHT, m_fBaseTransFrame, vRight);
		CLWGlobal::iteminfo->param(objID, LWIP_W_UP, m_fBaseTransFrame, vUp);
		CLWGlobal::iteminfo->param(objID, LWIP_W_FORWARD, m_fBaseTransFrame, vForward);

		node->m_vRotation[1] = -asin(vForward[1]);	// Pitch
		if(node->m_vRotation[1] != 0.0)
		{
			node->m_vRotation[0] = asin(vForward[0] / cos(node->m_vRotation[1]));	// Heading
			node->m_vRotation[2] = acos(vUp[1] / cos(node->m_vRotation[1]));	// Bank
		} else
		{
			node->m_vRotation[0] = asin(vForward[0]);	// Heading
			node->m_vRotation[2] = acos(vUp[1]);	// Bank
		}
		*/

		LWDVector	Pos, Rot;
		CLWGlobal::iteminfo->param(objID, LWIP_POSITION, m_fBaseTransFrame, Pos);
		CLWGlobal::iteminfo->param(objID, LWIP_ROTATION, m_fBaseTransFrame, Rot);

		node->m_vPosition[0] = (float) Pos[0];
		node->m_vPosition[1] = (float) Pos[1];
		node->m_vPosition[2] = (float) Pos[2];

		node->m_vRotation[0] = (float) Rot[0];
		node->m_vRotation[1] = (float) Rot[1];
		node->m_vRotation[2] = (float) Rot[2];
	}

	sprintf(buffer,"%i",objnum);
	strncpy(node->m_szNumber,buffer,10);
	node->m_nNumber = objnum;

	//*** Set the type of node and bone weight map info for bones
	node->m_ParentObject = NULL;
	strncpy(node->m_szWeightMap,"",MAX_OBJECT_NAME);
	if(CLWGlobal::iteminfo->type(objID) == LWI_OBJECT)
	{
		//*** Check if there are any polygons on the object
		if(CLWGlobal::objinfo->numPolygons(objID) == 0)
		{
			//*** NULL
			node->m_nType = NT_NULL;
		} else
		{
			//*** MESH
			node->m_nType = NT_MESH;
		}
	} else if(CLWGlobal::iteminfo->type(objID) == LWI_BONE)
	{
		//*** BONE
		node->m_nType = NT_BONE;

		//*** Copy the weight map name into the node
		if(CLWGlobal::boneinfo->weightMap(objID) != (char*)NULL)
		{
			strncpy(node->m_szWeightMap,CLWGlobal::boneinfo->weightMap(objID),MAX_OBJECT_NAME);
		} else
		{
			strncpy(node->m_szWeightMap,"",MAX_OBJECT_NAME);
		}

		//*** Walk up the hierarchy and find the bone's object
		LWItemID parentID = CLWGlobal::iteminfo->parent(objID);
		while( (CLWGlobal::iteminfo->type(parentID) != LWI_OBJECT) && (parentID != LWITEM_NULL) )
		{
			parentID = CLWGlobal::iteminfo->parent(parentID);
		}
		node->m_ParentObject = parentID;
	}

	m_vNodes.push_back(node);
}

//************************************************************
//*** GETNUMBEROFNODES()
//*** Returns the total number of nodes defined
//************************************************************
int CLWData::GetNumberOfNodes()
{
	return (m_vNodes.size());
}

//************************************************************
//*** GETNODENUMBER()
//*** Returns the node number based on the LWItemID
//************************************************************
int CLWData::GetNodeNumber(LWItemID objID)
{
	vector<NodeInfo*>::iterator itor;

	for(itor = m_vNodes.begin(); itor != m_vNodes.end(); ++itor)
	{
		if((*itor)->m_ID == objID)
		{
			return (*itor)->m_nNumber;
		}
	}

	return (-1);
}

//************************************************************
//*** GETNODEPARENTMESH()
//*** Returns the total number of nodes defined
//************************************************************
int CLWData::GetNodeParentMesh(int node)
{
	int i;
	int parentnum = -1;

	int numMeshes = GetObjectCount();
	for (i = 0 ; i < numMeshes ; i++)
	{
		LWItemID objID = m_vObjectList[i]->m_ID;
		if(objID == m_vNodes[node]->m_ID)
		{
			parentnum = i;
		}
	}

	return parentnum;
}

//************************************************************
//*** GETNODENAME()
//*** Returns the total number of nodes defined
//************************************************************
char* CLWData::GetNodeName(int node)
{
	return (m_vNodes[node]->m_szName);
}

//************************************************************
//*** GETNODEPARENT()
//*** Returns the parent node number
//************************************************************
int CLWData::GetNodeParent(int node)
{
	return (m_vNodes[node]->m_nParentNode);
}

//************************************************************
//*** GETNODEPOSITION()
//*** Returns the node's position vector
//************************************************************
float* CLWData::GetNodePosition(int node)
{
	return (m_vNodes[node]->m_vPosition);
}

//************************************************************
//*** GETNODEROTATION()
//*** Returns the node's position vector
//************************************************************
float* CLWData::GetNodeRotation(int node)
{
	return (m_vNodes[node]->m_vRotation);
}

//************************************************************
//*** DOESNODEEXIST()
//*** Returns true is the given object is already in the node list
//************************************************************
bool CLWData::DoesNodeExist(LWItemID objID)
{
	vector<NodeInfo*>::iterator itor;

	for(itor = m_vNodes.begin(); itor != m_vNodes.end(); ++itor)
	{
		if((*itor)->m_ID == objID)
		{
			return true;
		}
	}

	return false;
}

//************************************************************
//*** DEFINEROOTNODE()
//*** Returns if a root node should be defined
//************************************************************
bool CLWData::DefineRootNode()
{
	//*** Check if a Root node has already been defined
	if(m_nRootNode != -1)
	{
		return true;
	}

	//*** Check if m_nExportMeshAsNode has not been set, and if so,
	//*** then a root node is required
	if(m_nExportMeshAsNode == 0)
	{
		return true;
	}

	//*** If a collision bounding box or cylinder is required, then
	//*** so is  root node
	if( (m_nCollisionType == 1) || (m_nCollisionType == 2) )
	{
		return true;
	}

	return false;
}

//************************************************************
//*** FINDMATCHINGBONENODE()
//*** Returns the node number of the bone that has the matching parent object and weight map.
//*** Returns -1 if there is no matching bone node
//************************************************************
int CLWData::FindMatchingBoneNode(LWItemID objID, char* wmap)
{
	assert(m_vNodes.size() != 0);	// Not truely a problem, but we shouldn't call this function if the node list hasn't first been built

	vector<NodeInfo*>::iterator itor;

	for(itor = m_vNodes.begin(); itor != m_vNodes.end(); ++itor)
	{
		if((*itor)->m_nType == NT_BONE)
		{
			//*** Found a bone node, so check its parent object
			if((*itor)->m_ParentObject == objID)
			{
				//*** OK, so we match the parent object.  Now onto the weight map
				if(strcmp((*itor)->m_szWeightMap, wmap) == 0)
				{
					return (*itor)->m_nNumber;
				}
			}
		}
	}

	return (-1);
}

//************************************************************
//*** GETANIMSEQUNCECOUNT()
//*** Returns the number of defined animation sequences
//************************************************************
int CLWData::GetAnimSequenceCount()
{
	int count = 0;
	vector<AnimInfo*>::iterator itor;

	for(itor = m_vSequences.begin(); itor != m_vSequences.end(); ++itor)
	{
		++count;
	}

	return count;
}

//************************************************************
//*** NEWANIMSEQUENCE()
//*** Creates a new animation sequence
//************************************************************
int CLWData::NewAnimSequence()
{
	AnimInfo* ainfo = new AnimInfo;

	strcpy(ainfo->m_szName,"new");
	ainfo->m_nFPS = 30;
	strcpy(ainfo->m_szFPS,"30");
	ainfo->m_nStart = 0;
	strcpy(ainfo->m_szStart,"0");
	ainfo->m_nEnd = 0;
	strcpy(ainfo->m_szEnd,"0");
	ainfo->m_nCyclic = 0;
	ainfo->m_nThread = 0;
	strcpy(ainfo->m_szThread,"0");

	int num = m_vSequences.size();

	m_vSequences.push_back(ainfo);

	return num;
}

//************************************************************
//*** DELETEANIMSEQUENCE()
//*** Removes a given animation sequence from the list
//************************************************************
void CLWData::DeleteAnimSequence(int seq)
{
	vector<AnimInfo*> temp;	// For storing the new compact list
	vector<AnimInfo*>::iterator itor;
	int count = 0;

	for(itor = m_vSequences.begin(); itor != m_vSequences.end();)
	{
		if(count == seq)
		{
			//*** Delete this object
			delete (*itor);
			itor = m_vSequences.erase(itor);
		} else
		{
			temp.push_back(*itor);
			++itor;
		}

		++count;
	}

	//*** Now swap the temp vector with the original one, thus making a new compact
	//*** object list
	m_vSequences.swap(temp);
}

//************************************************************
//*** MoveAnimSequenceUp()
//*** Moves an animation sequence up the list, if possible
//************************************************************
void CLWData::MoveAnimSequenceUp(int seq)
{
	vector<AnimInfo*>::iterator itor;
	int count = 0;

	for(itor = m_vSequences.begin(); itor != m_vSequences.end(); ++itor)
	{
		if(count == seq)
		{
			//*** Found the sequence in the list.  Move it up if possible.
			if(count != 0)
			{
				AnimInfo* temp = m_vSequences[count];
				m_vSequences[count] = m_vSequences[(count-1)];
				m_vSequences[(count-1)] = temp;
				break;
			}
		}
		++count;
	}

}

//************************************************************
//*** MoveAnimSequenceDown()
//*** Moves an animation sequence down the list, if possible
//************************************************************
void CLWData::MoveAnimSequenceDown(int seq)
{
	vector<AnimInfo*>::iterator itor;
	int count = 0;

	for(itor = m_vSequences.begin(); itor != m_vSequences.end(); ++itor)
	{
		if(count == seq)
		{
			//*** Found the sequence in the list.  Move it down if possible.
			if(count != (GetAnimSequenceCount()-1))
			{
				AnimInfo* temp = m_vSequences[count];
				m_vSequences[count] = m_vSequences[(count+1)];
				m_vSequences[(count+1)] = temp;
				break;
			}
		}
		++count;
	}
}

//************************************************************
//*** GETFIRSTANIMSEQUENCENAME()
//*** Returns the name of the first anim seq that includes the given frame
//************************************************************
char* CLWData::GetFirstAnimSequenceName(int frame)
{
	vector<AnimInfo*>::iterator itor;

	//*** Find a sequence that contains the given frame.
	for(itor = m_vSequences.begin(); itor != m_vSequences.end(); ++itor)
	{
		if( (frame >= (*itor)->m_nStart) && (frame <= (*itor)->m_nEnd) )
		{
			return (*itor)->m_szName;
		}
	}

	return "No Sequence";
}

//************************************************************
//*** GETFIRSTANIMSEQUENCESTART()
//*** Returns the start frame of the first anim seq that includes the given frame
//************************************************************
char* CLWData::GetFirstAnimSequenceStart(int frame)
{
	vector<AnimInfo*>::iterator itor;

	//*** Find a sequence that contains the given frame.
	for(itor = m_vSequences.begin(); itor != m_vSequences.end(); ++itor)
	{
		if( (frame >= (*itor)->m_nStart) && (frame <= (*itor)->m_nEnd) )
		{
			return (*itor)->m_szStart;
		}
	}

	return "-";
}

//************************************************************
//*** GETFIRSTANIMSEQUENCEEND()
//*** Returns the start frame of the first anim seq that includes the given frame
//************************************************************
char* CLWData::GetFirstAnimSequenceEnd(int frame)
{
	vector<AnimInfo*>::iterator itor;

	//*** Find a sequence that contains the given frame.
	for(itor = m_vSequences.begin(); itor != m_vSequences.end(); ++itor)
	{
		if( (frame >= (*itor)->m_nStart) && (frame <= (*itor)->m_nEnd) )
		{
			return (*itor)->m_szEnd;
		}
	}

	return "-";
}

//************************************************************
//*** CLEARANIMSEQUENCE()
//*** Clears the animation sequence list
//************************************************************
void CLWData::ClearAnimSequence()
{
	vector<AnimInfo*>::iterator itor;

	//*** Delete the AnimInfo instances.
	for(itor = m_vSequences.begin(); itor != m_vSequences.end(); ++itor)
	{
		delete (*itor);
	}

	m_vSequences.clear();
}

//************************************************************
//*** CHECKPARAMETERS()
//*** Checks that all user setable parameters have appropriate values for
//*** an export.  True=go ahead, false=something wrong
//************************************************************
bool CLWData::CheckParameters()
{
	//*** Confirm that a file name has been given
	if(m_szFilename[0] == '\0')
	{
		CLWGlobal::msg->error("You must provide a DTS filename.","");
		return false;
	}

	//*** Check that if Mesh Collision is chosen that at least one object has been selected
	//*** to export for collision
	if(m_nCollisionType == CLWData::CT_Mesh)
	{
		if(GetFirstCollisionObject() == 0)
		{
			CLWGlobal::msg->error("When a collision type of Mesh is selected, an object must be selected to export as a collision object.","Please select an object and change its export type to collision, or change the collision type.");
			return false;
		}
	}

	return true;
}

//************************************************************
//*** SAVE()
//*** Saves the global and object list data
//************************************************************
int CLWData::Save( const LWSaveState *save )
{
	int nVersion = EX_VERSION;
	int nCount = 0;

	//*** Refresh the object list and their names
	RefreshObjectList();
	RefreshObjectNames();

	//*** Save the version of this plug-in
	LWSAVE_BEGIN( save, &idVersion[ 0 ], 1 );
		LWSAVE_I4( save, (long*) &nVersion, 1 );
	LWSAVE_END( save );

	//*** Advanced tab stuff

	//*** Save out m_nPolyPointType
	LWSAVE_I4( save, (long*) &m_nPolyPointType, 1 );

	if(nVersion >= 2)
	{
		//*** Save out m_fBaseTransFrame
		LWSAVE_FP( save, &m_fBaseTransFrame, 1 );

		//*** Save out m_nExportMeshAsNode
		LWSAVE_I4( save, (long*) &m_nExportMeshAsNode, 1 );

		//*** Save out m_szMeshNodeExt
		LWSAVE_STR( save, m_szMeshNodeExt);
	}

	//*** Global tab stuff

	//*** Save out m_nCollisionType
	LWSAVE_I4( save, (long*) &m_nCollisionType, 1 );

	//*** Save out m_nCollisionVisible
	LWSAVE_I4( save, (long*) &m_nCollisionVisible, 1 );

	//*** Save out m_fCollisionComplexity
	LWSAVE_FP( save, &m_fCollisionComplexity, 1 );

	//*** Save out m_nExportMaterials
	LWSAVE_I4( save, (long*) &m_nExportMaterials, 1 );

	//*** Save out m_szFilename
	LWSAVE_STR( save, m_szFilename);

	//*** Save out m_fScaleFactor
	LWSAVE_FP( save, &m_fScaleFactor, 1 );

	//*** Save out m_nMinPixelSize
	LWSAVE_I4( save, (long*) &m_nMinPixelSize, 1 );

	//*** Object based stuff

	//*** Save out each of the objects
	nCount = GetObjectCount();
	LWSAVE_I4( save, (long*) &nCount, 1 );

	if(nCount > 0)
	{
		for(int i=0; i < nCount; ++i)
		{
			//*** Save out m_ID
			LWSAVE_I4( save, (long*) &(m_vObjectList[i]->m_ID), 1 );

			//*** Save out m_szLWName
			LWSAVE_STR( save, m_vObjectList[i]->m_szLWName);

			//*** Save out m_szDTSName
			LWSAVE_STR( save, m_vObjectList[i]->m_szDTSName);

			//*** Save out m_nExportType
			LWSAVE_I4( save, (long*) &(m_vObjectList[i]->m_nExportType), 1 );

			//*** Save out m_nGlobalRotation
			LWSAVE_I4( save, (long*) &(m_vObjectList[i]->m_nGlobalRotation), 1 );

			//*** Save out m_nEncodedNormals
			LWSAVE_I4( save, (long*) &(m_vObjectList[i]->m_nEncodedNormals), 1 );
		}
	}

	//*** Save out the animation sequences
	if(nVersion >= 3)
	{
		//*** Save the 'Export animation sequence' flag
		LWSAVE_I4( save, (long*) &m_nExportAnimSequences, 1 );

		//*** Save the number of animation sequences
		nCount = GetAnimSequenceCount();
		LWSAVE_I4( save, (long*) &nCount, 1 );

		if(nCount > 0)
		{
			for(int i=0; i < nCount; ++i)
			{
				//*** Save out m_szName
				LWSAVE_STR( save, m_vSequences[i]->m_szName);

				//*** Save out m_nStart
				LWSAVE_I4( save, (long*) &(m_vSequences[i]->m_nStart), 1 );

				//*** Save out m_nEnd
				LWSAVE_I4( save, (long*) &(m_vSequences[i]->m_nEnd), 1 );

				//*** Save out m_nFPS
				LWSAVE_I4( save, (long*) &(m_vSequences[i]->m_nFPS), 1 );

				//*** Save out m_nCyclic
				LWSAVE_I4( save, (long*) &(m_vSequences[i]->m_nCyclic), 1 );

				//*** Save out m_nThread
				LWSAVE_I4( save, (long*) &(m_vSequences[i]->m_nThread), 1 );
			}
		}

	}

	return 1;
}

//************************************************************
//*** LOAD()
//*** Loads the global and object list data
//************************************************************
int CLWData::Load( const LWLoadState *load )
{
	int nVersion = 0;
	int nCount = 0;
	LWID id;

	//*** Load in the version of the plug-in that saved the data
	id = LWLOAD_FIND( load, idVersion );
	if(id)
	{
		LWLOAD_I4( load, (long*) &nVersion, 1 );
	} else
	{
		CLWGlobal::msg->info("Cannot determine version of DTS Exporter data","");
		return 1;
	}
	LWLOAD_END( load );

	//*** Advanced tab stuff

	//*** Load in m_nPolyPointType
	LWLOAD_I4( load, (long*) &m_nPolyPointType, 1 );

	if(nVersion >= 2)
	{
		//*** Load in m_fBaseTransFrame
		LWLOAD_FP( load, &m_fBaseTransFrame, 1 );

		//*** Load in m_nExportMeshAsNode
		LWLOAD_I4( load, (long*) &m_nExportMeshAsNode, 1 );

		//*** Load in m_szMeshNodeExt
		LWLOAD_STR( load, m_szMeshNodeExt, MAX_OBJECT_NAME);
	}

	//*** Global tab stuff

	//*** Load in m_nCollisionType
	LWLOAD_I4( load, (long*) &m_nCollisionType, 1 );

	//*** Load in m_nCollisionVisible
	LWLOAD_I4( load, (long*) &m_nCollisionVisible, 1 );

	//*** Load in m_fCollisionComplexity
	LWLOAD_FP( load, &m_fCollisionComplexity, 1 );

	//*** Load in m_nExportMaterials
	LWLOAD_I4( load, (long*) &m_nExportMaterials, 1 );

	//*** Load in m_szFilename
	LWLOAD_STR( load, m_szFilename, MAX_FILE_PATH);

	//*** Load in m_fScaleFactor
	LWLOAD_FP( load, &m_fScaleFactor, 1 );

	//*** Load in m_nMinPixelSize
	LWLOAD_I4( load, (long*) &m_nMinPixelSize, 1 );

	//*** Object based stuff

	//*** Load in the number of objects
	LWLOAD_I4( load, (long*) &nCount, 1 );

	//*** Load in each of the objects
	if(nCount > 0)
	{
		for(int i=0; i<nCount; ++i)
		{
			ObjectInfo* node = new ObjectInfo;

			//*** Load in m_ID
			LWLOAD_I4( load, (long*) &(node->m_ID), 1 );

			//*** Load in m_szLWName
			LWLOAD_STR( load, node->m_szLWName, MAX_OBJECT_NAME);

			//*** Load in m_szDTSName
			LWLOAD_STR( load, node->m_szDTSName, MAX_OBJECT_NAME);

			//*** Load in m_nExportType
			LWLOAD_I4( load, (long*) &(node->m_nExportType), 1 );

			//*** Load in m_nGlobalRotation
			LWLOAD_I4( load, (long*) &(node->m_nGlobalRotation), 1 );

			//*** Load in m_nEncodedNormals
			LWLOAD_I4( load, (long*) &(node->m_nEncodedNormals), 1 );

			//*** Now add the object to the list
			m_vObjectList.push_back(node);
		}
	}

	//*** Load in the animation sequences
	if(nVersion >= 3)
	{
		//*** Load the 'Export animation sequence' flag
		LWLOAD_I4( load, (long*) &m_nExportAnimSequences, 1 );

		//*** Load in the number of sequences
		LWLOAD_I4( load, (long*) &nCount, 1 );

		//*** Load in each sequence
		if(nCount > 0)
		{
			for(int i=0; i < nCount; ++i)
			{
				AnimInfo* seq = new AnimInfo;
				char tempbuffer[64];

				//*** Load in m_szName
				LWLOAD_STR( load, seq->m_szName, MAX_OBJECT_NAME);

				//*** Load in m_nStart
				LWLOAD_I4( load, (long*) &(seq->m_nStart), 1 );
				sprintf(tempbuffer,"%i",seq->m_nStart);
				strncpy(seq->m_szStart,tempbuffer,64);

				//*** Load in m_nEnd
				LWLOAD_I4( load, (long*) &(seq->m_nEnd), 1 );
				sprintf(tempbuffer,"%i",seq->m_nEnd);
				strncpy(seq->m_szEnd,tempbuffer,64);

				//*** Load in m_nFPS
				LWLOAD_I4( load, (long*) &(seq->m_nFPS), 1 );
				sprintf(tempbuffer,"%i",seq->m_nFPS);
				strncpy(seq->m_szFPS,tempbuffer,64);

				//*** Load in m_nCyclic
				LWLOAD_I4( load, (long*) &(seq->m_nCyclic), 1 );

				//*** Load in m_nThread
				LWLOAD_I4( load, (long*) &(seq->m_nThread), 1 );
				sprintf(tempbuffer,"%i",seq->m_nThread);
				strncpy(seq->m_szThread,tempbuffer,64);

				//*** Add the sequence to the list
				m_vSequences.push_back(seq);
			}
		}
	}

	return 1;
}

//************************************************************
//*** FINDFIRSTINSTANCE()
//*** Within m_vList, find the first CLWData instance
//************************************************************
CLWData* CLWData::FindFirstInstance()
{
	vector<CLWData*>::iterator itor;

	//*** Find the CLWData class
	for(itor = m_vList.begin(); itor != m_vList.end(); ++itor)
	{
		if((*itor) != NULL)
		{
			return (*itor);
		}
	}

	return (CLWData*)NULL;
}

vector<CLWData*> CLWData::m_vList;
