/*	LWDATA.H
**
**	Defines the CLWData class that contains all of the configuration data
**
**	Revision History:
**	February 11, 2002	David Wyand		Created file
**	March 12, 2002		David Wyand		Added code to support the building of a global node list.
**	April 4, 2002		David Wyand		Changed the node structure to support weight maps for
**										bones.
**	April 4, 2002		David Wyand		Increased the EX_VERSION to 3 to account for
**										animation sequences.
**	April 9, 2002		David Wyand		Added support for a global list of CLWData classes
*/

#ifndef __LWDATA_H__
#define __LWDATA_H__

//*** Includes
#include <vector>
#include <lwsurf.h>

//*** Definitions
#define MAX_OBJECT_NAME		256
#define MAX_FILE_PATH		1024

#define EX_VERSION			3

#define ID_VERS  LWID_( 'V','E','R','S' )
#define ID_PPTP  LWID_( 'P','P','T','P' )
#define ID_COLT  LWID_( 'C','O','L','T' )
#define ID_COLV  LWID_( 'C','O','L','V' )
#define ID_COLC  LWID_( 'C','O','L','C' )
#define ID_EXPM  LWID_( 'E','X','P','M' )
#define ID_FILE  LWID_( 'F','I','L','E' )
#define ID_SFAC  LWID_( 'S','F','A','C' )
#define ID_MNPS  LWID_( 'M','N','P','S' )

#define ID_OBJI  LWID_( 'O','B','J','I' )

#define ID_OIDX  LWID_( 'O','I','D','X' )
#define ID_OLWN  LWID_( 'O','L','W','N' )
#define ID_ODTS  LWID_( 'O','D','T','S' )
#define ID_OEXT  LWID_( 'O','E','X','T' )
#define ID_OROT  LWID_( 'O','R','O','T' )
#define ID_ONOR  LWID_( 'O','N','O','R' )

//*** Globals
static LWBlockIdent idVersion[] = {
	ID_VERS, "Version",
	0,0
};

static LWBlockIdent idGlobalData[] = {
	ID_PPTP, "PolyPointType",
	ID_COLT, "CollisionType",
	ID_COLV, "CollisionVisible",
	ID_COLC, "CollisionComplexity",
	ID_EXPM, "ExportMaterials",
	ID_FILE, "DTSFileName",
	ID_SFAC, "ScaleFactor",
	ID_MNPS, "MinPixelSize",
	ID_OBJI, "ObjectInfoList",
	0
};

static LWBlockIdent idObjectInfoHeader[] = {
	ID_OBJI, "ObjectInfoList",
	0
};

static LWBlockIdent idObjectInfoData[] = {
	ID_OIDX, "LWItemID",
	ID_OLWN, "LWName",
	ID_ODTS, "DTSName",
	ID_OEXT, "ExportType",
	ID_OROT, "GlobalRotation",
	ID_ONOR, "EncodedNormals",
	0
};

using namespace std;

//*** CLWData class definition
class CLWData
{
public:
	//*** Object information structure
	typedef struct st_ObjectInfo {
		LWItemID	m_ID;							// LW object ID
		char		m_szLWName[MAX_OBJECT_NAME];	// LW name for this object (stored for convienence)
		char		m_szDTSName[MAX_OBJECT_NAME];	// Custom name for this object
		int			m_nExportType;					// How to export 0=None, 1=Mesh, 2=Collision
		int			m_nGlobalRotation;				// 0=Normal, 1=Billboard, 2=BillboardZ
		int			m_nEncodedNormals;				// 0=No, 1=Yes
	} ObjectInfo;

	//*** Node information structure
	typedef struct st_NodeInfo {
		LWItemID	m_ID;							// LW object ID
		char		m_szName[MAX_OBJECT_NAME];		// Name of this node
		int			m_nNumber;						// Absolute node number
		char		m_szNumber[10];					// Absolute node number in string format
		LWItemID	m_Parent;						// LW ID of the parent
		char		m_szParent[MAX_OBJECT_NAME];	// Name of this node
		int			m_nParentNode;					// Node number of the parent. -1 = None
		float		m_vPosition[3];					// Position of the node (relative to parent)
		float		m_vRotation[3];					// Rotation of the node (relative to parent)
		int			m_nType;						// Type of object 0=Null, 1=Bone, 2=Mesh
		LWItemID	m_ParentObject;					// Parent object for a bone node
		char		m_szWeightMap[MAX_OBJECT_NAME];	// Name of the weight map for a bone
	} NodeInfo;

	//*** Animation sequence information structure
	typedef struct st_AnimInfo {
		char		m_szName[MAX_OBJECT_NAME];		// Anim sequence name
		int			m_nStart;						// Sequence start frame
		char		m_szStart[64];					// Start frame as text
		int			m_nEnd;							// End frame
		char		m_szEnd[64];					// End frame as text
		int			m_nFPS;							// Sequence frames per second
		char		m_szFPS[64];					// FPS as text
		int			m_nCyclic;						// 0=Not cyclic, 1=Yes
		int			m_nThread;						// Animation thread for this sequence
		char		m_szThread[64];					// Thread as text
	} AnimInfo;

	enum ExportType
	{
		ET_None = 0,
		ET_Mesh,
		ET_Collision,
		ET_Node,
	};

	enum CollisionType
	{
		CT_None = 0,
		CT_BBox,
		CT_BCylinder,
		CT_Mesh,
	};

	enum NodeType
	{
		NT_NULL = 0,
		NT_BONE,
		NT_MESH,
	};

	//*** Functions
public:
	CLWData();
	virtual ~CLWData();

	void RefreshObjectList();				// There may be some new objects to add to our list
	ObjectInfo* BuildNode(LWItemID id);		// Build out a single object to add to the list
	void DeleteObject(LWItemID id);			// Delete a single object
	void ChangeObjectID(LWItemID oldid, LWItemID newid);	// Change the ID of an object
	void RefreshObjectNames();				// Rebuilds the names of the objects, in case they have changed
	int GetObjectCount();					// Returns the number of objects
	int GetObjectExportType(LWItemID id);	// Returns the export type of an object
	bool IsInObjectList(LWItemID id);		// true = object is in the object list

	LWItemID GetObjectByName(char* name);	// Returns ID of the object given its name
	char* GetObjectUserName(LWItemID id);	// Returns the DTS or LW name of an object as appropriate
	LWItemID GetFirstCollisionObject();		// Returns ID of the first collision object, or 0 if none.  This is a temp function until full support for multiple col objects is implemented

	void AddSurface(LWSurfaceID surf);		// Add surface to list
	int GetSurfaceIndex(LWSurfaceID surf);	// Look up the index of a surface
	void ClearSurfaces();					// Clear out the surface list

	void BuildNodeList();					// Build a list of all nodes
	void ClearNodeList();					// Clear the node list
	void BuildNodeChildList(int &objnum, LWItemID objID, bool localcoords);	// Build the list of children nodes
	void AddNodeToList(int objnum, LWItemID objID, bool isobject, bool localcoords);	// Build a node and add it to the list
	bool ConditionalAddObjectToNodeList(int &objnum, LWItemID objID, bool localcoords);	// Will add an object to the node list and return true if the object was added
	int GetNumberOfNodes();					// Returns the number of nodes
	int GetNodeNumber(LWItemID objID);		// Obtain the node number based on the LWItemID
	int GetNodeParentMesh(int node);		// Obtain the parent's absolute number
	char* GetNodeName(int node);			// Obtain the name of the node
	int GetNodeParent(int node);			// Obtain the parent node number for the given node
	float* GetNodePosition(int node);		// Obtain the node's position vector
	float* GetNodeRotation(int node);		// Obtain the node's rotation vector
	bool DoesNodeExist(LWItemID objID);		// Check if node has been added to list already
	bool DefineRootNode();					// Returns if a root node should be (has been) built
	int FindMatchingBoneNode(LWItemID objID, char* wmap);	// Finds a bone's node based on the parent object's ID and the weight map name

	int GetAnimSequenceCount();				// Returns the number of defined animation sequences
	int NewAnimSequence();					// Creates a new animation sequence
	void DeleteAnimSequence(int seq);		// Deletes an animation sequence
	void MoveAnimSequenceUp(int seq);		// Moves an animation sequence up the list
	void MoveAnimSequenceDown(int seq);		// Moves an animation sequence down
	char* GetFirstAnimSequenceName(int frame);	// Returns the name of the first anim seq that includes the given frame
	char* GetFirstAnimSequenceStart(int frame);	// Returns the start frame of the first anim seq that includes the given frame
	char* GetFirstAnimSequenceEnd(int frame);	// Returns the end frame of the first anim seq that includes the given frame
	void ClearAnimSequence();				// Clears the anim sequence list

	bool CheckParameters();					// Returns true if all parameters are set accordingly and it is possible to do the export

	int Save( const LWSaveState *save );
	int Load( const LWLoadState *load );

	static CLWData* FindFirstInstance();	// Find within m_vList the first CLWData class

	//*** Variables
public:
	int					m_nLastSelected;	// The last item in the mesh list the user selected
	vector<ObjectInfo*>	m_vObjectList;		// List of objects in scene and their data
	int					m_nPolyPointType;	// 0=original, 1=transformed
	float				m_fBaseTransFrame;	// Frame from which node positions/rotations should be taken from
	int					m_nExportMeshAsNode;	// Export meshes as nodes 0=No, 1=Yes
	char				m_szMeshNodeExt[MAX_OBJECT_NAME];	// Extension added to a mesh-node

	int					m_nCollisionType;	// Collision mesh type: 0=None, 1=Bounding Box, 2=Bounding Cylinder, 3=Mesh
	int					m_nCollisionVisible;	// Display collision meshes 0=No, 1=Yes
	float				m_fCollisionComplexity;	// How complex a mesh should we make (for bounding cylinder mostly)
	int					m_nExportMaterials;	// Include material in the file 0=No, 1=Yes
	char				m_szFilename[MAX_FILE_PATH];	// DTS file name from the user
	float				m_fScaleFactor;		// Over-all scale factor for the model
	int					m_nMinPixelSize;	// Minimum pixel size
	vector<NodeInfo*>	m_vNodes;

//	bool				m_bRootNodeDefined;	// true if a Root node has been built
	int					m_nRootNode;		// Points to the Root Node.  -1 if not defined

	int					m_nExportAnimSequences;	// Include anim sequences in the file 0=No, 1=Yes
	vector<AnimInfo*>	m_vSequences;		// List of animation sequences

	static vector<CLWData*>	m_vList;		// Static list holding pointers to all defined CLWData classes

protected:
	vector<LWSurfaceID>	m_vSurfaces;

};


#endif // __LWDATA_H__
