/*	LWEXPORTERSHADER.CPP
**
**	Functions for the shader plug-in used by the DTS exporter plug-in.  This plug-in will
**	allow the user to set surface specific parameters for the DTS file, such as additive
**	or subtractive attributes for transparent surfaces.
**
**	Revision History:
**	March 29, 2002		David Wyand		Created file
**	April 5, 2002		David Wyand		Modified ESCopy() to correctly copy the class members
**										from the source instance to the destination instance.
**	April 26, 2002		David Wyand		Removed 'static' function declarations based on the
**										Mac Code Warrior build by C J Silverio.
*/

//*** Includes
#include <lwserver.h>			// all plug-ins need this
#include <lwshader.h>			// for shader functions
#include <lwpanel.h>			// for "classic" panels
#include <lwhost.h>				// for the LWMessageFuncs global
#include "LWGlobal.h"
#include "LWESData.h"
#include "GUIExportShader.h"
#include "LWExporterShader.h"

//*** Global variables
GlobalFunc *g_ShaderGlobal;				// Pointer to all global functions
LWPanelFuncs *shaderPanf;				//* the panel functions
LWPanelID shaderPanel = 0;				//* the panel

//*** Prototypes
extern "C"
{
XCALL_(const char*)		ESDescribe(LWInstance);
XCALL_(LWInstance)		ESCreate(void *,void *,LWError *);
XCALL_(void)			ESDestroy(LWInstance);
XCALL_(LWError)			ESCopy(LWInstance,LWInstance);
XCALL_(LWError)			ESLoad(LWInstance,const LWLoadState *);
XCALL_(LWError)			ESSave(LWInstance,const LWSaveState *);
XCALL_(LWError)			ESInit(LWInstance, int);
XCALL_(void)			ESCleanUp(LWInstance);
XCALL_(LWError)			ESNewTime(LWInstance, LWFrame, LWTime);
XCALL_(void)			ESEvaluate(LWInstance,LWShaderAccess *);
XCALL_(unsigned int)	ESFlags(LWInstance);
XCALL_(LWError)			ESOptions(LWInstance inst);
}


//************************************************************
//*** ACTIVATE()
//*** Layout calls this when the user selects the plugin from the list
//*** of shaders.
//************************************************************
extern "C"
{
//************
XCALL_(int) ExpShaderActivate( long version, GlobalFunc *global, LWShaderHandler *local, void *serverData )
{
	//*** Check that Layout is of the correct version
	if ( version != LWSHADER_VERSION )
		return AFUNC_BADVERSION;

	//*** Pass along pointers to the different plug-in functions (generic)
	if(local->inst)
	{
		local->inst->create     = ESCreate;
		local->inst->destroy    = ESDestroy;
		local->inst->load       = ESLoad;
		local->inst->save       = ESSave;
		local->inst->copy       = ESCopy;
		local->inst->descln     = ESDescribe;
	}

	//*** Pass along pointers to the different plug-in render functions (shader)
	if(local->rend)
	{
		local->rend->init     = ESInit;
		local->rend->cleanup  = ESCleanUp;
		local->rend->newTime  = ESNewTime;
	}
	local->evaluate         = ESEvaluate;
	local->flags            = ESFlags;

	//*** Get the panels functions
	shaderPanf = (LWPanelFuncs *) (*global)( LWPANELFUNCS_GLOBAL, GFUSE_TRANSIENT );
	if ( !shaderPanf )
		return AFUNC_BADGLOBAL;

	//*** Initalize the CLWGlobal static class
	if( CLWGlobal::Initialize(global) == false)
	{
		return AFUNC_BADGLOBAL;
	}

	g_ShaderGlobal = global;

	return AFUNC_OK;
}

//************************************************************
//*** ESCREATE()
//*** Called when an instance of the plug-in has been created
//************************************************************
XCALL_(LWInstance) ESCreate(void *priv, void *context, LWError *err)
{
	CLWESData*	mydata;

	mydata = new CLWESData;

	mydata->m_ID = (LWSurfaceID) context;

	return(mydata);
}

//************************************************************
//*** ESDESTROY()
//*** Called when an instance of the plug-in has been destroyed
//************************************************************
XCALL_(void) ESDestroy(LWInstance inst)
{
	CLWESData*	mydata;

	mydata = (CLWESData*) inst;

	if(mydata)
	{
		delete mydata;
	}

}

//************************************************************
//*** ESCOPY()
//*** Called when an instance of the plug-in needs to be copied
//************************************************************
XCALL_(LWError) ESCopy(LWInstance d1,LWInstance d2)
{
	CLWESData*	source;
	CLWESData*	dest;

	source = (CLWESData*) d2;
	dest = (CLWESData*) d1;

	dest->m_ID = source->m_ID;
	dest->m_nMipMapZeroBorder = source->m_nMipMapZeroBorder;
	dest->m_nNoMipMap = source->m_nNoMipMap;
	dest->m_nTransparentType = source->m_nTransparentType;

    return((LWError) NULL);
}

//************************************************************
//*** ESLOAD()
//*** Called to load in the plug-in data from a scene
//************************************************************
XCALL_(LWError) ESLoad(LWInstance inst,const LWLoadState *lState)
{
	CLWESData*	mydata;

	mydata = (CLWESData*) inst;

	mydata->Load(lState);

    return((LWError) NULL);
}

//************************************************************
//*** ESSAVE()
//*** Called to save the plug-in data to a scene
//************************************************************
XCALL_(LWError) ESSave(LWInstance inst,const LWSaveState *sState)
{
	CLWESData*	mydata;

	mydata = (CLWESData*) inst;

	mydata->Save(sState);

    return((LWError) NULL);
}

//************************************************************
//*** ESDESCRIBE()
//*** Called when Layout needs to display the name of the plug-in to the user
//************************************************************
XCALL_(const char *) ESDescribe(LWInstance inst)
{
    return("DTS Exporter Material Attributes");
}

//************************************************************
//*** ESINIT()
//*** Called prior to rendering to perform any setup
//************************************************************
XCALL_( LWError ) ESInit( LWInstance inst, int mode )
{

	return (LWError) NULL;
}

//************************************************************
//*** ESCLEANUP()
//*** Called after rendering to perform any cleanup
//************************************************************
XCALL_( void ) ESCleanUp( LWInstance inst )
{
	return;
}

//************************************************************
//*** ESNEWTIME()
//*** Called at the start of each sampling pass
//************************************************************
XCALL_( LWError ) ESNewTime( LWInstance inst, LWFrame f, LWTime t )
{
	return (LWError) NULL;
}

//************************************************************
//*** ESFLAGS()
//*** Called to tell layout what types of events we should be sent.
//************************************************************
XCALL_(unsigned int) ESFlags(LWInstance inst)
{
	return(0);
}

//************************************************************
//*** ESEVALUATE()
//*** Where the actual rendering happens.
//************************************************************
XCALL_( void ) ESEvaluate( LWInstance inst, LWShaderAccess *sa )
{
}

} // extern "C"

//************************************************************
//*** INTERFACE()
//*** Layout calls this when it needs info about the GUI
//************************************************************
extern "C"
{
XCALL_(int) ExpShaderInterface(long version, GlobalFunc *global, void *local, void *serverData )
{
	LWInterface* mylocal;
	
	if ( version != LWINTERFACE_VERSION )
		return AFUNC_BADVERSION;

	mylocal = (LWInterface*) local;

	mylocal->panel   = NULL;
	mylocal->options = ESOptions;
	mylocal->command = 0; // (NULL);

	if(shaderPanf)
	{
		shaderPanf->globalFun = global;
	}

	return(AFUNC_OK);
}

//************************************************************
//*** ESOPTIONS()
//*** Layout calls this when the user opens the GUI
//************************************************************
XCALL_(LWError) ESOptions(LWInstance inst)
{
	CGUIExportShader exportshadergui;
	CLWESData* instData = (CLWESData*) inst;

	//*** Define the panel
	if( !( shaderPanel = PAN_CREATE( shaderPanf, CLWGlobal::GetPanelTitle() )))
	{
		CLWGlobal::msg->info("Cannot open the user interface!","");
		return 0;
	}

	//*** Open the panel
	if(shaderPanel)
	{
		//*** Create all of the controls
		exportshadergui.CreateExportShader(instData, shaderPanf, shaderPanel, 0, 0, 450, 200);

		//*** Display the panel
		shaderPanf->open( shaderPanel, PANF_BLOCKING );

		PAN_KILL( shaderPanf, shaderPanel );
	}

	return(AFUNC_OK);
}

} // extern "C"
