/*	LWIKITEMMOTION.CPP
**
**	Functions for the IK Item Motion Handler plug-in used by the DTS exporter plug-in.
**	It converts IK data into a custom channel that allows DTS animation sequences to
**	be built using IK animation.  There is no user interface as everything is keyed off
**	of an object's motion properties.
**
**	Revision History:
**	April 11, 2002		David Wyand		Created file
**	April 26, 2002		David Wyand		Removed 'static' function declarations based on the
**										Mac Code Warrior build by C J Silverio.
*/

//*** Includes
#include <stdio.h>
#include <lwserver.h>			// all plug-ins need this
#include <lwhandler.h>
#include <lwrender.h>
#include <lwmotion.h>
#include <lwpanel.h>			// for "classic" panels
#include <lwhost.h>				// for the LWMessageFuncs global
#include "LWGlobal.h"
#include "LWIKItemMotion.h"

//*** Global variables
GlobalFunc *g_IKItemMotionGlobal;			// Pointer to all global functions

//*** Prototypes
extern "C"
{
XCALL_(const char*)		IKIMDescribe(LWInstance);
XCALL_(LWInstance)		IKIMCreate(void *,void *,LWError *);
XCALL_(void)			IKIMDestroy(LWInstance);
XCALL_(LWError)			IKIMCopy(LWInstance,LWInstance);
XCALL_(LWError)			IKIMLoad(LWInstance,const LWLoadState *);
XCALL_(LWError)			IKIMSave(LWInstance,const LWSaveState *);
XCALL_(LWError)			IKIMInit(LWInstance, int);
XCALL_(void)			IKIMCleanUp(LWInstance);
XCALL_(LWError)			IKIMNewTime(LWInstance, LWFrame, LWTime);
XCALL_(void)			IKIMEvaluate(LWInstance,const LWItemMotionAccess *);
XCALL_(unsigned int)	IKIMFlags(LWInstance);
XCALL_(LWError)			IKIMOptions(LWInstance inst);
}


//************************************************************
//*** ACTIVATE()
//*** Layout calls this when the user selects the plugin from the list of
//*** custom object plug-ins.
//************************************************************
extern "C"
{
//************
XCALL_(int) IKItemMotionActivate( long version, GlobalFunc *global, LWItemMotionHandler *local, void *serverData )
{
	//*** Check that Layout is of the correct version
	if ( version != LWITEMMOTION_VERSION )
		return AFUNC_BADVERSION;

	//*** Pass along pointers to the different plug-in functions (generic)
	if(local->inst)
	{
		local->inst->create     = IKIMCreate;
		local->inst->destroy    = IKIMDestroy;
		local->inst->load       = IKIMLoad;
		local->inst->save       = IKIMSave;
		local->inst->copy       = IKIMCopy;
		local->inst->descln     = IKIMDescribe;
	}

	local->evaluate         = IKIMEvaluate;
	local->flags            = IKIMFlags;

	//*** Initalize the CLWGlobal static class
	if( CLWGlobal::Initialize(global) == false)
	{
		return AFUNC_BADGLOBAL;
	}

	g_IKItemMotionGlobal = global;

	return AFUNC_OK;
}

//************************************************************
//*** IKIMCREATE()
//*** Called when an instance of the plug-in has been created
//************************************************************
XCALL_(LWInstance) IKIMCreate(void *priv, void *context, LWError *err)
{
	return(context);
}

//************************************************************
//*** IKIMDESTROY()
//*** Called when an instance of the plug-in has been destroyed
//************************************************************
XCALL_(void) IKIMDestroy(LWInstance inst)
{

}

//************************************************************
//*** IKIMCOPY()
//*** Called when an instance of the plug-in needs to be copied
//************************************************************
XCALL_(LWError) IKIMCopy(LWInstance d1,LWInstance d2)
{
    return((LWError) NULL);
}

//************************************************************
//*** IKIMLOAD()
//*** Called to load in the plug-in data from a scene
//************************************************************
XCALL_(LWError) IKIMLoad(LWInstance inst,const LWLoadState *lState)
{
    return((LWError) NULL);
}

//************************************************************
//*** IKIMSAVE()
//*** Called to save the plug-in data to a scene
//************************************************************
XCALL_(LWError) IKIMSave(LWInstance inst,const LWSaveState *sState)
{
    return((LWError) NULL);
}

//************************************************************
//*** IKIMDESCRIBE()
//*** Called when Layout needs to display the name of the plug-in to the user
//************************************************************
XCALL_(const char *) IKIMDescribe(LWInstance inst)
{
    return("DTS Exporter IK Channel Creator");
}

//************************************************************
//*** IKIMFLAGS()
//*** Called to tell layout what types of events we should be sent.
//************************************************************
XCALL_(unsigned int) IKIMFlags(LWInstance inst)
{
	return(LWIMF_AFTERIK);
}

//************************************************************
//*** IKIMEVALUATE()
//*** Where the actual calculations happen.
//************************************************************
XCALL_( void ) IKIMEvaluate( LWInstance inst, const LWItemMotionAccess *ima )
{
	LWDVector	rot;
	int			motType[3];
	LWChanGroupID	chanGroup;
	LWEnvelopeID envid;

	//*** Obtain the item's rotation
	ima->getParam( LWIP_ROTATION, ima->time, rot );

	//*** Get the motion type and channel group for the item
	CLWGlobal::iteminfo->controller(inst, LWIP_ROTATION, motType);
	chanGroup = CLWGlobal::iteminfo->chanGroup(inst);

	//*** If Heading is based off of IK, then add a keyframe to the custom channel
	if(motType[0] == LWMOTCTL_IK)
	{
		envid = CLWGlobal::findEnv( chanGroup, "DTSIK.H" );
		if(!envid)
		{
			//*** Couldn't find the envelope for the channel, so create it
			envid = CLWGlobal::envfunc->create( chanGroup, "DTSIK.H", LWET_ANGLE );
		}

		//*** Create the key for this time
		if(envid)
		{
			LWEnvKeyframeID keyid = CLWGlobal::envfunc->createKey( envid, ima->time, rot[0] );
		}

	}

	//*** If Pitch is based off of IK, then add a keyframe to the custom channel
	if(motType[1] == LWMOTCTL_IK)
	{
		envid = CLWGlobal::findEnv( chanGroup, "DTSIK.P" );
		if(!envid)
		{
			//*** Couldn't find the envelope for the channel, so create it
			envid = CLWGlobal::envfunc->create( chanGroup, "DTSIK.P", LWET_ANGLE );
		}

		//*** Create the key for this time
		if(envid)
		{
			LWEnvKeyframeID keyid = CLWGlobal::envfunc->createKey( envid, ima->time, rot[1] );
		}

	}

	//*** If Bank is based off of IK, then add a keyframe to the custom channel
	if(motType[2] == LWMOTCTL_IK)
	{
		envid = CLWGlobal::findEnv( chanGroup, "DTSIK.B" );
		if(!envid)
		{
			//*** Couldn't find the envelope for the channel, so create it
			envid = CLWGlobal::envfunc->create( chanGroup, "DTSIK.B", LWET_ANGLE );
		}

		//*** Create the key for this time
		if(envid)
		{
			LWEnvKeyframeID keyid = CLWGlobal::envfunc->createKey( envid, ima->time, rot[2] );
		}

	}

}

} // extern "C"

//************************************************************
//*** INTERFACE()
//*** Layout calls this when it needs info about the GUI
//************************************************************
extern "C"
{
XCALL_(int) IKItemMotionInterface(long version, GlobalFunc *global, void *local, void *serverData )
{
	LWInterface* mylocal;
	
	if ( version != LWINTERFACE_VERSION )
		return AFUNC_BADVERSION;

	mylocal = (LWInterface*) local;

	mylocal->panel   = NULL;
	mylocal->options = IKIMOptions;
	mylocal->command = 0; // (NULL);

	return(AFUNC_OK);
}

//************************************************************
//*** IKIMOPTIONS()
//*** Layout calls this when the user opens the GUI
//************************************************************
XCALL_(LWError) IKIMOptions(LWInstance inst)
{
	CLWGlobal::msg->info("DTS Exporter IK Channel Creator","This plug-in will create a DTSIK channel for objects\nthat are animated using IK.");

	return(AFUNC_OK);
}

} // extern "C"
