/*	LWOBJECTDB.CPP
**
**	Supports the CLWObjectDB class.  See LWObjectDB.h for a history of
**	where this class came from.
**
**	Revision History:
**	February 18, 2002	David Wyand		Created file
**	April 6, 2002		David Wyand		Added a check to GetPointUV() to determine if a point
**										is indeed attached to a polygon.  If not, then default
**										uv coordinates are used.
**	April 7, 2002		David Wyand		Fixed a bug in GetObjectVMaps() that was causing the
**										wrong vmap to be used for an object.  It had to do with
**										referencing a block of the vmap list starting with the
**										first vmap when only certain ones within the list should
**										be referenced.  See the block beginning with:
**										for ( i = -1, vmapnum = 0; vmapnum < vmdb->nvmaps; vmapnum++ )
**	May 28, 2002		David Wyand		Added code to prevent the allocation of memory if
**										there are no vmaps for a given point in GetObjectVMaps().
*/

//*** Includes
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "LWGlobal.h"

#include "LWObjectDB.h"

//*** Defines
#define vecangle(a,b) (float)acos(dot(a,b))    /* a and b must be unit vectors */

//*** Function prototypes
static int pntScan( CLWObjectDB *odb, LWPntID id );
static int polScan( CLWObjectDB *odb, LWPolID id );

//*** Constructor
CLWObjectDB::CLWObjectDB(LWItemID itemID, GlobalFunc *global)
{
	m_pGlobal = global;
	m_bLoadWorked = false;
	
	m_ID = itemID;

	m_szFilename = NULL;
	m_nPoints = 0;
	m_nPolys = 0;
	m_nSurfaces = 0;
	m_nVertMaps = 0;
	m_pPoints = NULL;
	m_pVSort = NULL;
	m_nVSortKey;
	m_pPoly = NULL;
	m_pSurfs = NULL;
	m_pVMap = NULL;

	if(GetObjectInfo() == true)
	{
		m_bLoadWorked = true;
	} else
	{
		m_bLoadWorked = false;
	}
}

//*** Destructor
CLWObjectDB::~CLWObjectDB()
{
	FreeObjectInfo();
	m_bLoadWorked = false;

}

//*** Confirm that the object was loaded into the database
bool CLWObjectDB::isLoaded()
{
	return m_bLoadWorked;
}

//*** Place the object into the DB
bool CLWObjectDB::GetObjectInfo()
{
	LWObjectInfo *objinfo = NULL;
	LWMeshInfo *mesh = NULL;
	LWPntID ptid;
	const char *szName;
	int npts, npols, nverts, i, j, k, ok = 0;
	int temppnt;

	//*** Get the object info global
	objinfo = (LWObjectInfo*) m_pGlobal( LWOBJECTINFO_GLOBAL, GFUSE_TRANSIENT );
	if ( !objinfo )
	{
		CLWGlobal::msg->error( "Couldn't obtain object info global","");
		return false;
	}

	//*** Get the mesh info for the object
	mesh = objinfo->meshInfo( m_ID, 1 );
	if ( !mesh )
	{
		CLWGlobal::msg->error( "Couldn't obtain mesh info for", CLWGlobal::iteminfo->name( m_ID ));
		return false;
	}

	//*** Retrieve some info on the object
	szName = objinfo->filename( m_ID );
	m_szFilename = (char*) malloc( strlen( szName ) + 1 );
	if ( !m_szFilename )
	{
		CLWGlobal::msg->error( "Couldn't obtain file name for", CLWGlobal::iteminfo->name( m_ID ));
		goto Finish;
	}
	strcpy( m_szFilename, szName );

	//*** Allocate and init the points array
	npts = mesh->numPoints( mesh );
	m_pPoints = (DBPoint*) calloc( npts, sizeof( DBPoint ));
	if ( !m_pPoints )
	{
		CLWGlobal::msg->error( "Couldn't allocate points array for", CLWGlobal::iteminfo->name( m_ID ));
		goto Finish;
	}

	if ( mesh->scanPoints( mesh, (LWPntScanFunc*) pntScan, this ))
	{
		CLWGlobal::msg->error( "Couldn't scan points for", CLWGlobal::iteminfo->name( m_ID ));
		goto Finish;
	}

	//*** Allocate and init the polygons array
	npols = mesh->numPolygons( mesh );
	m_pPoly = (DBPolygon*) calloc( npols, sizeof( DBPolygon ));
	if ( !m_pPoly )
	{
		CLWGlobal::msg->error( "Couldn't allocate polygon array for", CLWGlobal::iteminfo->name( m_ID ));
		goto Finish;
	}

	if ( mesh->scanPolys( mesh, (LWPolScanFunc *) polScan, this ))
	{
		CLWGlobal::msg->error( "Couldn't scan polygons for", CLWGlobal::iteminfo->name( m_ID ));
		goto Finish;
	}

	//*** Get the vertices of each polygon
	for ( i = 0; i < npols; i++ )
	{
		nverts = mesh->polSize( mesh, m_pPoly[ i ].id );
		m_pPoly[ i ].v = (DBPolVert*) calloc( nverts, sizeof( DBPolVert ));
		if ( !m_pPoly[ i ].v )
		{
			CLWGlobal::msg->error( "Couldn't allocate vertex array for", CLWGlobal::iteminfo->name( m_ID ));
			goto Finish;
		}

		m_pPoly[ i ].nverts = nverts;
		for ( j = 0; j < nverts; j++ )
		{
			ptid = mesh->polVertex( mesh, m_pPoly[ i ].id, j );
			m_pPoly[ i ].v[ j ].index = FindVert( ptid );
		}
	}

	//*** Count the number of polygons per point
	for ( i = 0; i < npols; i++ )
	{
		for ( j = 0; j < m_pPoly[ i ].nverts; j++ )
		{
			++m_pPoints[ m_pPoly[ i ].v[ j ].index ].npols;
		}
	}

	//*** Allocate per-point polygon arrays
	for ( i = 0; i < npts; i++ )
	{
		if ( m_pPoints[ i ].npols == 0 )
		{
			continue;
		}
		
		m_pPoints[ i ].pol = (int*) calloc( m_pPoints[ i ].npols, sizeof( int ));
		if ( !m_pPoints[ i ].pol )
		{
			CLWGlobal::msg->error( "Couldn't allocate per-point polygon array for", CLWGlobal::iteminfo->name( m_ID ));
			goto Finish;
		}
		
		m_pPoints[ i ].npols = 0;
	}

	//*** Fill in polygon array for each point
	for ( i = 0; i < npols; i++ )
	{
		for ( j = 0; j < m_pPoly[ i ].nverts; j++ )
		{
			k = m_pPoly[ i ].v[ j ].index;
			m_pPoints[ k ].pol[ m_pPoints[ k ].npols ] = i;
			++m_pPoints[ k ].npols;
		}
	}

	//*** Get the position of each point
	for ( i = 0; i < npts; i++ ) {
		mesh->pntBasePos( mesh, m_pPoints[ i ].id, m_pPoints[ i ].pos[ 0 ] );
		mesh->pntOtherPos( mesh, m_pPoints[ i ].id, m_pPoints[ i ].pos[ 1 ] );
	}

   /* init the point search array */

   /* Ordinarily, you won't need this because you can look up points
      by their IDs.  Uncomment this if you do need to search by point
      position instead.  The second argument is 0 for base position
      and 1 for final position.

      if ( !initPointSearch( odb, 0 ))
         goto Finish;
   */

	//*** Calculate the normal of each polygon
	GetPolyNormals( 0 );
	GetPolyNormals( 1 );

	//*** Get the vmaps
	if ( !GetObjectVMaps( mesh ))
	{
		CLWGlobal::msg->error( "Couldn't obtain vmaps for", CLWGlobal::iteminfo->name( m_ID ));
		goto Finish;
	}

	//*** Get the surfaces
	if ( !GetObjectSurfs( mesh ))
	{
		CLWGlobal::msg->error( "Couldn't obtain surfaces for", CLWGlobal::iteminfo->name( m_ID ));
		goto Finish;
	}

	//*** Calculate vertex normals
	GetVertNormals( 0 );
	GetVertNormals( 1 );

	//*** done
	ok = 1;

Finish:
	if ( mesh->destroy )
	{
		mesh->destroy( mesh );
	}

	if ( !ok )
	{
		FreeObjectInfo();
		return false;
	}

	return true;
}

//*** Free all of the memory associated with the DB
void CLWObjectDB::FreeObjectInfo()
{
   int i;

	FreeObjectVMaps();
	FreeObjectSurfs();
	//+++ freePointSearch();

	if ( m_pPoints )
	{
		for ( i = 0; i < m_nPoints; i++ )
		{
			if ( m_pPoints[ i ].pol )
			{
				free( m_pPoints[ i ].pol );
			}
		}
		free( m_pPoints );
	}

	if ( m_pPoly )
	{
		for ( i = 0; i < m_nPolys; i++ )
		{
			if ( m_pPoly[ i ].v )
			{
				free( m_pPoly[ i ].v );
			}
		}
         free( m_pPoly );
	}
}

//*** Point scan callback to insert the point ID into the point array in
//*** ascending numerical order.
static int pntScan( CLWObjectDB *odb, LWPntID id )
{
	int j;

	j = odb->m_nPoints;

	if ( j == 0 )
	{
		odb->m_pPoints[ 0 ].id = id;
		++odb->m_nPoints;
		return 0;
	}

	while ( odb->m_pPoints[ j - 1 ].id > id )
	{
		odb->m_pPoints[ j ].id = odb->m_pPoints[ j - 1 ].id;
		--j;
		if ( j == 0 )
		{
			break;
		}
	}

	odb->m_pPoints[ j ].id = id;
	++odb->m_nPoints;

	return 0;
}

//*** Polygon scan callback to store the polygon ID
static int polScan( CLWObjectDB *odb, LWPolID id )
{
	odb->m_pPoly[ odb->m_nPolys ].id = id;
	++odb->m_nPolys;
	return 0;
}

//*** FindVert()
//*** Binary search the point array and return the array index for the given point ID
int CLWObjectDB::FindVert( LWPntID id )
{
	int lt = 0, rt = m_nPoints - 1, x;

	while ( rt >= lt )
	{
		x = ( lt + rt ) / 2;
		if ( id < m_pPoints[ x ].id )
		{
			rt = x - 1;
		} else
		{
			lt = x + 1;
		}
		if ( id == m_pPoints[ x ].id )
		{
			return x;
		}
	}
	return -1;
}

//*** GetPolyNormals()
//*** Calcualte the polygon normals.  LW's polygon normals are based on the
//*** first, second and last points in the vertex list.  The normal is the
//*** cross product of two vectors formed from there points.  The normal is
//*** undefined for one and tw-point polygons.
//*** The 'pointtype' param indicates which form of the vertex to work off of.
//*** Currently, 0=original vertex position; 1=tranformed vertex position.
void CLWObjectDB::GetPolyNormals( int pointtype )
{
   int j, k;
   LWFVector p1, p2, pn, v1, v2;

   for ( j = 0; j < m_nPolys; j++ ) {
      if ( m_pPoly[ j ].nverts < 3 ) continue;
      for ( k = 0; k < 3; k++ ) {
         p1[ k ] = m_pPoints[ m_pPoly[ j ].v[ 0 ].index ].pos[ pointtype ][ k ];
         p2[ k ] = m_pPoints[ m_pPoly[ j ].v[ 1 ].index ].pos[ pointtype ][ k ];
         pn[ k ] = m_pPoints[ m_pPoly[ j ].v[m_pPoly[ j ].nverts - 1 ].index ].pos[ pointtype ][ k ];
      }

      for ( k = 0; k < 3; k++ ) {
         v1[ k ] = p2[ k ] - p1[ k ];
         v2[ k ] = pn[ k ] - p1[ k ];
      }

      cross( v1, v2, m_pPoly[ j ].norm[ pointtype ] );
      normalize( m_pPoly[ j ].norm[ pointtype ] );
   }
}

//*** GetVertNormals()
//*** Calcualtes the vertex's normals.  For each vertex that belongs to a polygon,
//*** sum the normals of the polygons that share the point.  If the normals of the
//*** current and adjacent polygons form an angle greater than the max smoothing angle
//*** for the current polygon's surface, the normal of the adjacent polygon is
//*** excluded from the sum.
//*** The 'pointtype' param indicates which form of the vertex to work off of.
//*** Currently, 0=original vertex position; 1=tranformed vertex position.
void CLWObjectDB::GetVertNormals( int pointtype )
{
	int j, k, n, g, h, p;
	float a;

	for ( j = 0; j < m_nPolys; j++ )
	{
		for ( n = 0; n < m_pPoly[ j ].nverts; n++ )
		{
			for ( k = 0; k < 3; k++ )
			{
				m_pPoly[ j ].v[ n ].norm[ pointtype ][ k ] = m_pPoly[ j ].norm[ pointtype ][ k ];
			}

			if ( m_pSurfs[ m_pPoly[ j ].sindex ].sman <= 0 )
			{
				continue;
			}

			p = m_pPoly[ j ].v[ n ].index;

			for ( g = 0; g < m_pPoints[ p ].npols; g++ )
			{
				h = m_pPoints[ p ].pol[ g ];
				if ( h == j )
				{
					continue;
				}

				a = vecangle( m_pPoly[ j ].norm[ pointtype ], m_pPoly[ h ].norm[ pointtype ] );
				if ( a > m_pSurfs[ m_pPoly[ j ].sindex ].sman )
				{
					continue;
				}

				for ( k = 0; k < 3; k++ )
				{
					m_pPoly[ j ].v[ n ].norm[ pointtype ][ k ] += m_pPoly[ h ].norm[ pointtype ][ k ];
				}
			}

			normalize( m_pPoly[ j ].v[ n ].norm[ pointtype ] );
		}
	}
}

/*
======================================================================
getObjectVMaps()

Allocate and fill in structures describing the vertex maps applied to
the object.  Updates the ObjectDB and returns 1 if successful.  Backs
out of any changes to the ObjectDB and returns 0 if an error occurs.

Before calling this function, the ObjectDB must be prepared to receive
vmap data.  This means that the point array has been allocated and
filled in, and the references to vmap structures are pristine--all
counts are 0 and all pointers are NULL.

We need to work within a peculiarity of the plug-in API for vmaps.
We can't ask for only the vmaps for a particular object.  In order to
find the vmaps associated with an object, we need to ask, for each
point in the object, whether there's a vmap value for that point.
Only vmaps for which at least one point has a value are added to the
ObjectDB.
====================================================================== */

//*** GetObjectVMaps()
//*** Allocate and fill in the structures describing the vertex maps applied to the object.
//*** We need to work within a peculiarity of the plug-in API for vmaps.  It is not
//*** possible to find out the vmaps for a particular object.  Instead, we must ask for
//*** for the vmaps associated with each of the points in the object.  Because of this, only
//*** vmaps for which at least one point has a value are added to the database.
//*** This function returns 1 if successful, and 0 on an error.
int CLWObjectDB::GetObjectVMaps( LWMeshInfo *mesh )
{
	VertMapDB *vmdb;
	void *vmid;
	int i, j, k, n, ismapped, nvmaps, dim, *npts, ok = 0;
	float *val = NULL;


	//*** Get the list of vmaps in the scene
	vmdb = GetVertMapDB();
	if ( !vmdb )
	{
		return 1;
	}

	//*** Create an array for counting the points with each vmap
	npts = (int*) calloc( vmdb->nvmaps, sizeof( int ));
	if ( !npts )
	{
		char buffer[256];
		sprintf(buffer,"vmdb->nvmaps = %d",vmdb->nvmaps);
		CLWGlobal::msg->error( "Couldn't allocate vmap point count array", buffer);
		goto Finish;
	}

	//*** Count the object's vmaps and the number of points for each vmap
	for ( i = 0, nvmaps = 0; i < vmdb->nvmaps; i++ )
	{
		vmid = mesh->pntVLookup( mesh, vmdb->vmap[ i ].type, vmdb->vmap[ i ].name );

		if ( vmid )
		{
			dim = mesh->pntVSelect( mesh, vmid );
			if ( dim > 0 )
			{
				val = (float*) calloc( dim, sizeof( float ));
				if ( !val )
				{
					char buffer[256];
					sprintf(buffer,"dim = %d",dim);
					CLWGlobal::msg->error( "Couldn't allocate vmap point array", buffer);
					goto Finish;
				}
			} else
			{
				val = NULL;
			}

			for ( j = 0; j < m_nPoints; j++ )
			{
				ismapped = mesh->pntVGet( mesh, m_pPoints[ j ].id, val );
				if ( ismapped )
				{
					vmdb->vmap[ i ].valid = true;
					++npts[ i ];
				}
			}

			if ( npts[ i ] )
			{
				++nvmaps;
			}

			free( val );
		}
	}

	//*** No vmaps for this object?
	if ( nvmaps == 0 )
	{
		ok = 1;
		goto Finish;
	}

	//*** Allocate the vmap array
	m_nVertMaps = nvmaps;
	m_pVMap = (DBVMap*) calloc( nvmaps, sizeof( DBVMap ));
	if ( !m_pVMap )
	{
		char buffer[256];
		sprintf(buffer,"nvmaps = %d",nvmaps);
		CLWGlobal::msg->error( "Couldn't allocate vmap array", buffer);
		goto Finish;
	}

//	for ( i = 0; i < nvmaps; i++ )
	int vmapnum;
	for ( i = -1, vmapnum = 0; vmapnum < vmdb->nvmaps; vmapnum++ )
	{
		if(vmdb->vmap[ vmapnum ].valid != true)
		{
			continue;
		}
		++i;

		//*** Initialize the vmap info
		m_pVMap[ i ].name = (char*) malloc( strlen( vmdb->vmap[ vmapnum ].name ) + 1 );
		if ( !m_pVMap[ i ].name )
		{
			char buffer[256];
			sprintf(buffer,"strlen(vmdb->vmap[ vmapnum ].name)+1 = %d",(strlen( vmdb->vmap[ vmapnum ].name ) + 1));
			CLWGlobal::msg->error( "Couldn't allocate vmap name", buffer);
			goto Finish;
		}

		strcpy( m_pVMap[ i ].name, vmdb->vmap[ vmapnum ].name );
		m_pVMap[ i ].type = vmdb->vmap[ vmapnum ].type;
		m_pVMap[ i ].dim = vmdb->vmap[ vmapnum ].dim;
		m_pVMap[ i ].nverts = npts[ vmapnum ];

		//*** Allocate the point index array
		m_pVMap[ i ].vindex = (int*) calloc( npts[ vmapnum ], sizeof( int ));
		if ( !m_pVMap[ i ].vindex )
		{
			char buffer[256];
			sprintf(buffer,"npts[ vmapnum ] = %d",npts[ vmapnum ]);
			CLWGlobal::msg->error( "Couldn't allocate vmap point index array", buffer);
			goto Finish;
		}

		//*** Allocate the value arrays
		if ( vmdb->vmap[ vmapnum ].dim > 0 )
		{
			m_pVMap[ i ].val = (float**) calloc( vmdb->vmap[ vmapnum ].dim, sizeof( float * ));
			if ( !m_pVMap[ i ].val )
			{
				char buffer[256];
				sprintf(buffer,"vmdb->vmap[ vmapnum ].dim = %d",vmdb->vmap[ vmapnum ].dim);
				CLWGlobal::msg->error( "Couldn't allocate vmap value array", buffer);
				goto Finish;
			}
			for ( k = 0; k < vmdb->vmap[ vmapnum ].dim; k++ )
			{
				m_pVMap[ i ].val[ k ] = (float*) calloc( npts[ vmapnum ], sizeof( float ));
				if ( !m_pVMap[ i ].val[ k ] )
				{
					char buffer[256];
					sprintf(buffer,"npts[ vmapnum ] = %d",npts[ vmapnum ]);
					CLWGlobal::msg->error( "Couldn't allocate vmap value subarray", buffer);
					goto Finish;
				}
			}
		}

		//*** Fill in the point index and value arrays
		vmid = mesh->pntVLookup( mesh, vmdb->vmap[ vmapnum ].type, vmdb->vmap[ vmapnum ].name );
		if ( vmid )
		{
			dim = mesh->pntVSelect( mesh, vmid );
			if ( dim > 0 )
			{
				val = (float*) calloc( dim, sizeof( float ));
				if ( !val )
				{
					char buffer[256];
					sprintf(buffer,"dim = %d",dim);
					CLWGlobal::msg->error( "Couldn't allocate vmap value", buffer);
					goto Finish;
				}
			} else
			{
				val = NULL;
			}

			for ( j = 0, n = 0; j < m_nPoints; j++ )
			{
				ismapped = mesh->pntVGet( mesh, m_pPoints[ j ].id, val );
				if ( ismapped )
				{
					m_pVMap[ i ].vindex[ n ] = j;
					for ( k = 0; k < dim; k++ )
					{
						m_pVMap[ i ].val[ k ][ n ] = val[ k ];
					}
					++n;
				}
			}
			free( val );
		}
	}

	//*** Count the number of vmap values for each point
	for ( i = 0; i < nvmaps; i++ )
	{
		for ( j = 0; j < m_pVMap[ i ].nverts; j++ )
		{
			++m_pPoints[ m_pVMap[ i ].vindex[ j ]].nvmaps;
		}
	}

	//*** Allocate vmap references for each mapped point
	for ( i = 0; i < m_nPoints; i++ )
	{
		//*** Skip if there is no vmap for this point
		if(m_pPoints[ i ].nvmaps == 0)
		{
			continue;
		}

		m_pPoints[ i ].vm = (DBVMapPt*) calloc( m_pPoints[ i ].nvmaps, sizeof( DBVMapPt ));
		if ( !m_pPoints[ i ].vm )
		{
			char buffer[256];
			sprintf(buffer,"m_pPoints[ i ].nvmaps = %d",m_pPoints[ i ].nvmaps);
			CLWGlobal::msg->error( "Couldn't allocate vmap reference", buffer);
			goto Finish;
		}
		m_pPoints[ i ].nvmaps = 0;
	}

	//*** Fill in vmap references for each mapped point
	for ( i = 0; i < nvmaps; i++ )
	{
		for ( j = 0; j < m_pVMap[ i ].nverts; j++ )
		{
			n = m_pVMap[ i ].vindex[ j ];
			k = m_pPoints[ n ].nvmaps;
			m_pPoints[ n ].vm[ k ].vmap = &m_pVMap[ i ];
			m_pPoints[ n ].vm[ k ].index = j;
			++m_pPoints[ n ].nvmaps;
		}
	}

	//*** Success
   ok = 1;

Finish:
	FreeVertMapDB( vmdb );
	if ( npts )
	{
		free( npts );
	}
	if ( !ok )
	{
		FreeObjectVMaps();
	}

	return ok;
}

//*** GetVertMapDB()
//*** Fills in a structure describing the vertex maps in a scene.  This is
//*** used by GetObjectVMaps().
CLWObjectDB::VertMapDB* CLWObjectDB::GetVertMapDB()
{
	VertMapDB *vmdb;
	LWObjectFuncs *objf;
	const char *name;
	int j, nvmaps;


	objf = (LWObjectFuncs*) m_pGlobal( LWOBJECTFUNCS_GLOBAL, GFUSE_TRANSIENT );
	if ( !objf )
	{
		return NULL;
	}

	//*** Count vmaps of all types
	nvmaps = objf->numVMaps( 0 );

	if ( !nvmaps )
	{
		return NULL;
	}

	//**** Allocate the VertMapDB
	vmdb = (VertMapDB*) calloc( 1, sizeof( VertMapDB ));
	if ( !vmdb )
	{
		return NULL;
	}
	vmdb->nvmaps = nvmaps;

	//*** Allocate the vmap array
	vmdb->vmap = (DBVMapRec*) calloc( nvmaps, sizeof( DBVMapRec ));
	if ( !vmdb->vmap )
	{
		free( vmdb );
		return NULL;
	}

	//*** Fill in the vmap array
	for ( j = 0; j < nvmaps; j++ )
	{
		vmdb->vmap[ j ].valid = false;	// Is VMap valid for the object ie: points use it?
		vmdb->vmap[ j ].type = objf->vmapType( j );
		vmdb->vmap[ j ].dim = objf->vmapDim( 0, j );
		name = objf->vmapName( 0, j );
		vmdb->vmap[ j ].name = (char*) malloc( strlen( name ) + 1 );
		if ( !vmdb->vmap[ j ].name )
		{
			FreeVertMapDB( vmdb );
			return NULL;
		}
		strcpy( vmdb->vmap[ j ].name, name );
	}

	return vmdb;
}

//*** FreeVertMapDB()
//*** Frees the VerMapDB created by GetVertMapDB()
void CLWObjectDB::FreeVertMapDB( VertMapDB *vmdb )
{
   int i;

   if ( vmdb ) {
      for ( i = 0; i < vmdb->nvmaps; i++ )
         if ( vmdb->vmap[ i ].name )
            free( vmdb->vmap[ i ].name );
      if ( vmdb->vmap )
         free( vmdb->vmap );
      free( vmdb );
   }
}

//*** FreeObjectVMaps()
//*** Frees memory allocated by GetObjectVMaps()
void CLWObjectDB::FreeObjectVMaps()
{
   int i, j;

	if ( m_pVMap )
	{
		for ( i = 0; i < m_nVertMaps; i++ )
		{
            if ( m_pVMap[ i ].name )
			{
               free( m_pVMap[ i ].name );
			}
            if ( m_pVMap[ i ].vindex )
			{
               free( m_pVMap[ i ].vindex );
			}
            if ( m_pVMap[ i ].val )
			{
               for ( j = 0; j < m_pVMap[ i ].dim; j++ )
			   {
                  if ( m_pVMap[ i ].val[ j ] )
				  {
                     free( m_pVMap[ i ].val[ j ] );
				  }
			   }
               free( m_pVMap[ i ].val );
            }
		}
		free( m_pVMap );
		m_pVMap = NULL;
		m_nVertMaps = 0;
	}

	for ( i = 0; i < m_nPoints; i++ )
	{
		m_pPoints[ i ].nvmaps = 0;
		if ( m_pPoints[ i ].vm )
		{
			free( m_pPoints[ i ].vm );
			m_pPoints[ i ].vm = NULL;
		}
	}
}

//*** GetObjectSurfs()
//*** Build the structures describing the surfaces applied to the object.
//*** Returns 1 if successful, 0 for an error.
int CLWObjectDB::GetObjectSurfs( LWMeshInfo *mesh )
{
	LWSurfaceFuncs *surff;
	LWTextureFuncs *txtrf;
	LWSurfaceID *surfid;
	LWTextureID tex;
	const char *tag;
	double *dval;
	int i;

	//*** Get the surface ID array
	surff = (LWSurfaceFuncs*) m_pGlobal( LWSURFACEFUNCS_GLOBAL, GFUSE_TRANSIENT );
	if ( !surff )
	{
		return 0;
	}
	surfid = surff->byObject( m_szFilename );
	if ( !surfid )
	{
		return 0;
	}

	//*** Get the pointer to the Texture Functions global
	txtrf = (LWTextureFuncs*) m_pGlobal( LWTEXTUREFUNCS_GLOBAL, GFUSE_TRANSIENT );
	if ( !txtrf )
	{
		return 0;
	}

	//*** Count the surface IDs and alloc the surface array
	for ( m_nSurfaces = 0; ; m_nSurfaces++ )
	{
		if ( !surfid[ m_nSurfaces ] ) break;
	}
	m_pSurfs = (DBSurface*) calloc( m_nSurfaces, sizeof( DBSurface ));
	if ( !m_pSurfs )
	{
		m_nSurfaces = 0;
		return 0;
	}

	//*** Fill in the surface parameters
	for ( i = 0; i < m_nSurfaces; i++ )
	{
		m_pSurfs[ i ].id = surfid[ i ];
		tag = surff->name( surfid[ i ] );
		m_pSurfs[ i ].name = (char*) malloc( strlen( tag ) + 1 );
		if ( !m_pSurfs[ i ].name )
		{
			FreeObjectSurfs();
			return 0;
		}
		strcpy( m_pSurfs[ i ].name, tag );

		dval = surff->getFlt( surfid[ i ], SURF_COLR );
		m_pSurfs[ i ].colr[ 0 ] = ( float ) dval[ 0 ];
		m_pSurfs[ i ].colr[ 1 ] = ( float ) dval[ 1 ];
		m_pSurfs[ i ].colr[ 2 ] = ( float ) dval[ 2 ];

		dval = surff->getFlt( surfid[ i ], SURF_LUMI );
		m_pSurfs[ i ].lumi = ( float ) *dval;

		dval = surff->getFlt( surfid[ i ], SURF_DIFF );
		m_pSurfs[ i ].diff = ( float ) *dval;

		dval = surff->getFlt( surfid[ i ], SURF_SPEC );
		m_pSurfs[ i ].spec = ( float ) *dval;

		dval = surff->getFlt( surfid[ i ], SURF_REFL );
		m_pSurfs[ i ].refl = ( float ) *dval;

		dval = surff->getFlt( surfid[ i ], SURF_TRAN );
		m_pSurfs[ i ].tran = ( float ) *dval;

		dval = surff->getFlt( surfid[ i ], SURF_TRNL );
		m_pSurfs[ i ].trnl = ( float ) *dval;

		dval = surff->getFlt( surfid[ i ], SURF_RIND );
		m_pSurfs[ i ].rind = ( float ) *dval;

		dval = surff->getFlt( surfid[ i ], SURF_BUMP );
		m_pSurfs[ i ].bump = ( float ) *dval;

		dval = surff->getFlt( surfid[ i ], SURF_GLOS );
		m_pSurfs[ i ].glos = ( float ) *dval;

		dval = surff->getFlt( surfid[ i ], SURF_SHRP );
		m_pSurfs[ i ].shrp = ( float ) *dval;

		dval = surff->getFlt( surfid[ i ], SURF_SMAN );
		m_pSurfs[ i ].sman = ( float ) *dval;

		dval = surff->getFlt( surfid[ i ], SURF_RSAN );
		m_pSurfs[ i ].rsan = ( float ) *dval;

		dval = surff->getFlt( surfid[ i ], SURF_TSAN );
		m_pSurfs[ i ].tsan = ( float ) *dval;

		dval = surff->getFlt( surfid[ i ], SURF_CLRF );
		m_pSurfs[ i ].clrf = ( float ) *dval;

		dval = surff->getFlt( surfid[ i ], SURF_CLRH );
		m_pSurfs[ i ].clrh = ( float ) *dval;

		dval = surff->getFlt( surfid[ i ], SURF_ADTR );
		m_pSurfs[ i ].adtr = ( float ) *dval;

		dval = surff->getFlt( surfid[ i ], SURF_AVAL );
		m_pSurfs[ i ].aval = ( float ) *dval;

		dval = surff->getFlt( surfid[ i ], SURF_GVAL );
		m_pSurfs[ i ].gval = ( float ) *dval;

		dval = surff->getFlt( surfid[ i ], SURF_LSIZ );
		m_pSurfs[ i ].lsiz = ( float ) *dval;

		dval = surff->getFlt( surfid[ i ], SURF_LCOL );
		m_pSurfs[ i ].lcol[ 0 ] = ( float ) dval[ 0 ];
		m_pSurfs[ i ].lcol[ 1 ] = ( float ) dval[ 1 ];
		m_pSurfs[ i ].lcol[ 2 ] = ( float ) dval[ 2 ];

		m_pSurfs[ i ].alph = surff->getInt( surfid[ i ], SURF_ALPH );
		m_pSurfs[ i ].rfop = surff->getInt( surfid[ i ], SURF_RFOP );
		m_pSurfs[ i ].trop = surff->getInt( surfid[ i ], SURF_TROP );
		m_pSurfs[ i ].side = surff->getInt( surfid[ i ], SURF_SIDE );
		m_pSurfs[ i ].glow = surff->getInt( surfid[ i ], SURF_GLOW );
		m_pSurfs[ i ].line = surff->getInt( surfid[ i ], SURF_LINE );

		m_pSurfs[ i ].rimg = surff->getImg( surfid[ i ], SURF_RIMG );
		m_pSurfs[ i ].timg = surff->getImg( surfid[ i ], SURF_TIMG );

		//*** Get the texture ID for the surface (TODO: Colour channel only for now!)
		m_pSurfs[ i ].colrproj = -1;	// Default to no projection type for the colour channel
		tex = surff->getTex( surfid[i], SURF_COLR );
		if(tex)
		{
			//*** This surface has a texture on the colour channel, so process it
			LWTLayerID tlayer = txtrf->firstLayer( tex );
			int type = txtrf->layerType( tlayer );
			if ( type == TLT_IMAGE )
			{
				//*** We have an image mappaed (as opposed to a procedural)
				//*** Obtain the type of projection
				txtrf->getParam( tlayer, TXTAG_PROJ, &(m_pSurfs[ i ].colrproj) );
				m_pSurfs[ i ].colrtlayer = tlayer;
			}
		}

	}

	//*** Find surface index for each polygon
	for ( i = 0; i < m_nPolys; i++ )
	{
		tag = mesh->polTag( mesh, m_pPoly[ i ].id, LWPTAG_SURF );
		m_pPoly[ i ].sindex = FindSurf( tag );
	}

	return 1;
}

//*** FreeObjectSurfs()
//*** Free memory allocated by GetObjectSurfs()
void CLWObjectDB::FreeObjectSurfs()
{
	int i;

	if ( m_pSurfs ) {
		for ( i = 0; i < m_nSurfaces; i++ )
		{
			if ( m_pSurfs[ i ].name )
			{
				free( m_pSurfs[ i ].name );
			}
		}

		free( m_pSurfs );
		m_nSurfaces = 0;
	}
}

/*
======================================================================
findSurf()

Linear search the surface array and return the array index with the
matching surface name.
====================================================================== */

//*** FindSurf()
//*** Searches through the surface array and returns the array index based
//*** on the matching name.
int CLWObjectDB::FindSurf( const char *tag )
{
	int i;

	for ( i = 0; i < m_nSurfaces; i++ )
	{
		if ( !strcmp( tag, m_pSurfs[ i ].name ))
		{
			return i;
		}
	}

	return -1;
}

//*** GetPointUV()
//*** Get the UV corrdinates of a point given its index.  Returns a 2 dimensional
//*** array containing the u and v.  If this point does not have a projection, then
//*** (1.0, 1.0) are returned.
void CLWObjectDB::GetPointUV(int pointindex, int polypointtype, double* uv)
{
	LWMeshInfo		*mesh;
	LWTextureFuncs	*txtrf;

	txtrf = (LWTextureFuncs*) m_pGlobal( LWTEXTUREFUNCS_GLOBAL, GFUSE_TRANSIENT );
	if(!txtrf)
	{
		CLWGlobal::msg->info("Could not initialize Texture Functions!","");
		uv[0] = 1.0;
		uv[1] = 1.0;
		return;
	}

	//*** Get the surface on the first polygon that this point belongs to.  Then
	//*** determine the surface projection type.  First make sure that this point
	//*** actually belongs to a polygon
	if(m_pPoints[pointindex].pol != NULL)
	{
		int polygon = m_pPoints[pointindex].pol[0];
		if(m_pSurfs)
		{
			DBSurface surf = m_pSurfs[m_pPoly[polygon].sindex];
			if(surf.colrproj != -1)
			{
				//*** OK, we can now deal with the surface, and determine the UV of
				//*** the first texture encountered.
				if ( surf.colrproj == TXPRJ_UVMAP )
				{
					//*** This point has been UV mapped
					void *vmap;
					float fuv[2];

					txtrf->getParam( surf.colrtlayer, TXTAG_VMAP, &vmap );

					if(vmap)
					{
						//*** Get the mesh info for the object
						mesh = CLWGlobal::objinfo->meshInfo( m_ID, 1 );
						if ( !mesh )
						{
							//*** None available
							uv[0] = 1.0;
							uv[1] = 1.0;
							return;
						}

						//pntVSet, pntVGet and pntVPGet functions
						//+++edit->pointVSet( edit->state, vmap, 0, NULL );
						mesh->pntVSelect(mesh, vmap);
						//+++edit->pointVEval( edit->state, pntID, polID, uv );
						mesh->pntVGet(mesh, m_pPoints[pointindex].id, fuv);

						uv[0] = fuv[0];
						uv[1] = fuv[1];

						//*** Cleanup the mesh object
						if ( mesh->destroy )
						{
							mesh->destroy( mesh );
						}

					} else
					{
CLWGlobal::msg->info("Cannot obtain vmap pointer based on surf.colrtlayer","");
						uv[0] = 1.0;
						uv[1] = 1.0;
					}

					return;
				} else
				{
					//*** This point has been projection mapped
					//double* pointpos = (double*) m_pPoints[pointindex].pos[0];
					double pointpos[3];
					pointpos[0] = m_pPoints[pointindex].pos[polypointtype][0];
					pointpos[1] = m_pPoints[pointindex].pos[polypointtype][1];
					pointpos[2] = m_pPoints[pointindex].pos[polypointtype][2];

					txtrf->evaluateUV( surf.colrtlayer, 0, 0, pointpos, pointpos, uv );
				
					//char tempbuff[256];
					//sprintf(tempbuff,"Point Index: %i x: %f y: %f z: %f u: %f v: %f",pointindex,pointpos[0],pointpos[1],pointpos[2],uv[0],uv[1]);
					//CLWGlobal::msg->info("GetPointUV",tempbuff);

					return;
				}
			}
		}
	}

	//*** Default values
	uv[0] = 1.0;
	uv[1] = 1.0;
}

//************************************************************
//*** GetPointVMapName()
//*** Retrieve the name of the first weight map of a given point
//************************************************************
char* CLWObjectDB::GetPointFirstWeightMap(int pointindex)
{
	int nummaps;
	int i;

	nummaps = m_pPoints[pointindex].nvmaps;

	if(nummaps == 0)
	{
		return NULL;
	}

	for(i=0; i<nummaps; ++i)
	{
		//*** Check if this is a weight map
		DBVMap* vmap = m_pPoints[pointindex].vm[i].vmap;

		if(vmap->type == LWVMAP_WGHT)
		{
			//*** Return the weight map's name
			return (vmap->name);
		}
	}

	return NULL;

}
