/*	LWOBJECTDB.H
**
**	Defines the CLWObjectDB class that provides access to a given LW object.
**	This is based on the OBJECTDB generic plug-in by Ernie Wright (thanks Ernie!)
**	For the original generic plug-in, visit: mywebpages.comcast.net/erniew/lwsdk/sample/scenscan/
**
**	Revision History:
**	February 18, 2002	David Wyand		Created file
*/

#ifndef __LWOBJECTDB_H__
#define __LWOBJECTDB_H__

//*** Includes
#include <math.h>
#include <lwserver.h>
#include <lwrender.h>
#include <lwmeshes.h>
#include <lwsurf.h>
#include <lwtxtr.h>

//*** Class definition
class CLWObjectDB
{
	//*** Vertex map database record definition
	typedef struct st_DBVMapRec {
		char		*name;
		LWID		type;
		int			dim;
		bool		valid;
	} DBVMapRec;

	//*** Vertex map database definition
	typedef struct st_VertMapDB {
		int			nvmaps;
		DBVMapRec	*vmap;
	} VertMapDB;

	//*** Vertex map point value definition
	typedef struct st_DBVMapVal {
		char		*name;			// name
		LWID		type;			// type of data (e.g. UVs, weights)
		int			dim;			// dimension (number of values per point)
		int			nverts;			// number of vertices
		int			*vindex;		// array of vertex indexes
		float		**val;			// 2D array of vmap values
	} DBVMap;

	//*** Vertex map point structure definition
	typedef struct st_DBVMapPt {
		DBVMap		*vmap;			// where the values are stored
		int			index;			// offset into the vmap value arrays
	} DBVMapPt;

	//*** Vertex structure definition
	typedef struct st_DBPoint {
		LWPntID		id;				// Layout point ID
		LWFVector	pos[ 2 ];		// initial and final position
		int			npols;			// number of polygons sharing the point
		int			*pol;			// array of polygon indexes
		int			nvmaps;			// number of vmap values
		DBVMapPt	*vm;			// array of vmap values
	} DBPoint;

	//*** Polygon vertex structure definition
	typedef struct st_DBPolVert {
		int			index;			// index into the point array
		LWFVector	norm[ 2 ];		// initial and final normals
		} DBPolVert;

	//*** Polygon structure definition
	typedef struct st_DBPolygon {
		LWPolID		id;				// Layout polygon ID
		int			sindex;			// surface index
		LWFVector	norm[ 2 ];		// initial and final normals
		int			nverts;			// number of vertices
		DBPolVert	*v;				// vertex array
	} DBPolygon;

	//*** Surface (material) structure definition
	typedef struct st_DBSurface {
		
		enum TextureType {
			IMAGE = TLT_IMAGE,
		};

		enum TextureChannel {
			TEXCHANNEL1 = 0,
			TEXCHANNEL2,
			TEXCHANNEL3,
			TEXCHANNEL4,
		};
		
		LWSurfaceID	id;				// surface ID
		char		*name;			// surface name
		float		colr[ 3 ];		// color
		float		lumi;			// luminosity
		float		diff;			// diffuse level
		float		spec;			// specularity
		float		refl;			// reflectivity
		float		tran;			// transparency
		float		trnl;			// translucency
		float		rind;			// index of refraction
		float		bump;			// bump
		float		glos;			// glossiness
		float		shrp;			// diffuse sharpness
		float		sman;			// max smoothing angle (radians)
		float		rsan;			// reflection seam angle
		float		tsan;			// refraction seam angle
		float		clrf;			// color filter
		float		clrh;			// color highlight
		float		adtr;			// additive transparency
		float		aval;			// alpha value
		float		gval;			// glow value
		float		lcol[ 3 ];		// line color
		float		lsiz;			// line size
		int			alph;			// alpha options
		int			rfop;			// reflection options
		int			trop;			// refraction options
		int			side;			// sidedness
		int			glow;			// glow
		int			line;			// render outlines
		LWImageID	rimg;			// reflection image
		LWImageID	timg;			// refraction image
		LWTLayerID	colrtlayer;		// first texture layer for colour channel
		int			colrproj;		// colour channel projection type
	} DBSurface;

	//*** Functions
public:
	CLWObjectDB(LWItemID itemID, GlobalFunc *global);
	virtual ~CLWObjectDB();

	bool	isLoaded();			// Did the object load into the database OK?

	int		FindVert( LWPntID id );	// Find the vertex index given a point ID
	void	GetPointUV(int pointindex, int polypointtype, double* uv); // Get the texture UV for a point
	char*	GetPointFirstWeightMap(int pointindex);	// Get the name of the VMap for a point

protected:
	bool	GetObjectInfo();					// Retrieve the object from LW Layout
	void	FreeObjectInfo();					// Free memory in DB
	void	GetPolyNormals( int pointtype );	// Retrieve polygon normals
	void	GetVertNormals( int pointtype );	// Retrieve vertex normals based on polys
	int		GetObjectVMaps( LWMeshInfo *mesh );	// Retrieve the vmaps for the object
	VertMapDB* GetVertMapDB();					// Retrieve all vmaps in the scene
	void	FreeVertMapDB( VertMapDB *vmdb );	// Frees memory for a VertMapDB structure
	void	FreeObjectVMaps();					// Frees memory for GetObjectVMaps()
	int		GetObjectSurfs( LWMeshInfo *mesh );	// Retrieve surface info for the polys
	void	FreeObjectSurfs();					// Frees memory for GetObjectSurfs()
	int		FindSurf( const char *tag );		// Find a surface index based on its name

	//*** Support functions
	inline float dot( LWFVector a, LWFVector b )
	{
		return a[ 0 ] * b[ 0 ] + a[ 1 ] * b[ 1 ] + a[ 2 ] * b[ 2 ];
	}

	inline void cross( LWFVector a, LWFVector b, LWFVector c )
	{
		c[ 0 ] = a[ 1 ] * b[ 2 ] - a[ 2 ] * b[ 1 ];
		c[ 1 ] = a[ 2 ] * b[ 0 ] - a[ 0 ] * b[ 2 ];
		c[ 2 ] = a[ 0 ] * b[ 1 ] - a[ 1 ] * b[ 0 ];
	}

	inline void normalize( LWFVector v )
	{
		float r;

		r = ( float ) sqrt( dot( v, v ));
		if ( r > 0 )
		{
			v[ 0 ] /= r;
			v[ 1 ] /= r;
			v[ 2 ] /= r;
		}
	}

	//*** Variables
public:
	char		*m_szFilename;	// object filename
	LWItemID	m_ID;			// Layout item ID
	int			m_nPoints;		// number of points
	int			m_nPolys;		// number of polygons
	int			m_nSurfaces;	// number of surfaces
	int			m_nVertMaps;	// number of vertex maps
	DBPoint		*m_pPoints;		// point array
	int			*m_pVSort;		// point indexes sorted by point position
	int			m_nVSortKey;	// coordinate for position sort
	DBPolygon	*m_pPoly;		// polygon array
	DBSurface	*m_pSurfs;		// surface array
	DBVMap		*m_pVMap;		// vmap array

protected:
	bool		m_bLoadWorked;	// Did the object load into the DB OK?
	GlobalFunc	*m_pGlobal;

};


#endif // __LWOBJECTDB_H__
