using System;
using System.Collections.Generic;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Audio;
using Microsoft.Xna.Framework.Components;
using Microsoft.Xna.Framework.Graphics;
using Microsoft.Xna.Framework.Input;
using Microsoft.Xna.Framework.Storage;
using XNAExtras; // For bitmap font

// GamePadXNA
// v1.1 released September 6, 2006
// Displays all states for up to four XInput controllers.  Written in C# for XNA Game
// Studio Express.
//
// Created by David Wyand
// www.gnometech.com
//
// Bitmap font code provided by Gary Kacmarcik
// blogs.msdn.com/garykac/default.aspx
// 
// Directions:
// Load up GamePadXNA.sln into Visual C# 2005 Express with XNA Game Studio Express.
// Compile and run.  A window will open displaying the various states for four Xinput 
// controllers which at the time this was written are the XBox 360 controllers.  Controllers
// may be added or removed from the system and their new state will be reflected.
//
// Release Notes:
// v1.0
// - Initial release.
//
// v1.1
// - Added graphics for each of the controllers.
// - The ESC key will close the program.


namespace GamePadXNA
{
    /// <summary>
    /// This is the main type for your game
    /// </summary>
    partial class GamePadXNA : Microsoft.Xna.Framework.Game
    {
        private BitmapFont mFont;
        private GamePadState[] mGamePadStates;
        private SpriteBatch mSpriteBatch;
        private Texture2D[] mGamePadActive;
        private Texture2D mGamePadInactive;


        public GamePadXNA()
        {
            mGamePadStates = new GamePadState[4];
            mGamePadActive = new Texture2D[4];

            InitializeComponent();
        }

        protected override void OnStarting()
        {
            base.OnStarting();
            graphics.GraphicsDevice.DeviceReset += new EventHandler(GraphicsDevice_DeviceReset);

            // Windows specific
            Window.Title = "GamePadXNA by David Wyand";

            // Create the font
            mFont = new BitmapFont("contents/fonts/", "arial.xml");

            LoadResources();
        }

        void GraphicsDevice_DeviceReset(object sender, EventArgs e)
        {
            LoadResources();
        }

        private void LoadResources()
        {
            // Reset the font when the graphic device changes
            mFont.Reset(graphics.GraphicsDevice);

            // Load in the game pad graphics
            mSpriteBatch = new SpriteBatch(graphics.GraphicsDevice);
            mGamePadActive[0] = Texture2D.FromFile(graphics.GraphicsDevice, "contents/images/controller1.dds");
            mGamePadActive[1] = Texture2D.FromFile(graphics.GraphicsDevice, "contents/images/controller2.dds");
            mGamePadActive[2] = Texture2D.FromFile(graphics.GraphicsDevice, "contents/images/controller3.dds");
            mGamePadActive[3] = Texture2D.FromFile(graphics.GraphicsDevice, "contents/images/controller4.dds");
            mGamePadInactive = Texture2D.FromFile(graphics.GraphicsDevice, "contents/images/controllerNone.dds");
        }

        protected bool checkExitKey()
        {
            // Has the ESC key been pressed?
            KeyboardState ks = Keyboard.GetState();
            if (ks.IsKeyDown(Keys.Escape))
            {
                Exit();
                return true;
            }

            return false;
        }

        protected override void Update()
        {
            // The time since Update was called last
            float elapsed = (float)ElapsedTime.TotalSeconds;

            if (!checkExitKey())
            {
                // Update the game pad states
                UpdateGamePad();

                // Let the GameComponents update
                UpdateComponents();
            }
        }

        protected void UpdateGamePad()
        {
            mGamePadStates[0] = GamePad.GetState(PlayerIndex.One);
            mGamePadStates[1] = GamePad.GetState(PlayerIndex.Two);
            mGamePadStates[2] = GamePad.GetState(PlayerIndex.Three);
            mGamePadStates[3] = GamePad.GetState(PlayerIndex.Four);
        }

        protected override void Draw()
        {
            // Make sure we have a valid device
            if (!graphics.EnsureDevice())
                return;

            graphics.GraphicsDevice.Clear(Color.CadetBlue);
            graphics.GraphicsDevice.BeginScene();

            // Draw the game pad states
            int fontheight = mFont.LineHeight;
            int posX = 140;
            int posY = 10;
            int margin = 4;
            int[] blockStart = { 10, 130, 250, 370 };

            // Draw the images
            mSpriteBatch.Begin(SpriteBlendMode.AlphaBlend);
            for (int i = 0; i < 4; ++i)
            {
                Vector2 pos = new Vector2(0, blockStart[i]-2);
                if (mGamePadStates[i].IsConnected)
                {
                    mSpriteBatch.Draw(mGamePadActive[i], pos, Color.White);
                }
                else
                {
                    mSpriteBatch.Draw(mGamePadInactive, pos, Color.White);
                }
            }
            mSpriteBatch.End();

            // Draw the text
            for (int i = 0; i < 4; ++i)
            {
                posY = blockStart[i];
                mFont.DrawString(22, posY, Color.White, "Game Pad " + (i+1));
                if (mGamePadStates[i].IsConnected)
                {
                    mFont.DrawString(44, posY+87, Color.LightGreen, "Active");
                }
                else
                {
                    mFont.DrawString(18, posY + 87, Color.DarkRed, "Disconnected");
                }

                posY += fontheight;
                mFont.DrawString(posX, posY, Color.White, "Buttons: ");
                if (mGamePadStates[i].IsConnected)
                {
                    drawButtons(mGamePadStates[i]);
                }

                posY += fontheight + margin;
                mFont.DrawString(posX, posY, "Left Stick: ");
                if (mGamePadStates[i].IsConnected)
                {
                    drawThumbStick(mGamePadStates[i].ThumbSticks.Left);
                }

                posY += fontheight + margin;
                mFont.DrawString(posX, posY, "Right Stick: ");
                if (mGamePadStates[i].IsConnected)
                {
                    drawThumbStick(mGamePadStates[i].ThumbSticks.Right);
                }

                posY += fontheight + margin;
                mFont.DrawString(posX, posY, "Triggers: ");
                if (mGamePadStates[i].IsConnected)
                {
                    drawTriggers(mGamePadStates[i]);
                }

                posY += fontheight + margin*3;
            }

            // Let the GameComponents draw
            DrawComponents();

            graphics.GraphicsDevice.EndScene();
            graphics.GraphicsDevice.Present();
        }

        protected void drawButtons(GamePadState state)
        {
            if (state.Buttons.A == ButtonState.Pressed)
            {
                mFont.DrawString("A ");
            }

            if (state.Buttons.B == ButtonState.Pressed)
            {
                mFont.DrawString("B ");
            }

            if (state.Buttons.X == ButtonState.Pressed)
            {
                mFont.DrawString("X ");
            }

            if (state.Buttons.Y == ButtonState.Pressed)
            {
                mFont.DrawString("Y ");
            }

            if (state.Buttons.Start == ButtonState.Pressed)
            {
                mFont.DrawString("START ");
            }

            if (state.Buttons.Back == ButtonState.Pressed)
            {
                mFont.DrawString("BACK ");
            }

            if (state.Buttons.LeftShoulder == ButtonState.Pressed)
            {
                mFont.DrawString("LB ");
            }

            if (state.Buttons.RightShoulder == ButtonState.Pressed)
            {
                mFont.DrawString("RB ");
            }

            if (state.Buttons.LeftStick == ButtonState.Pressed)
            {
                mFont.DrawString("LS ");
            }

            if (state.Buttons.RightStick == ButtonState.Pressed)
            {
                mFont.DrawString("RS ");
            }

            if (state.DPad.Left == ButtonState.Pressed)
            {
                mFont.DrawString("DPAD_LEFT ");
            }

            if (state.DPad.Right == ButtonState.Pressed)
            {
                mFont.DrawString("DPAD_RIGHT ");
            }

            if (state.DPad.Up == ButtonState.Pressed)
            {
                mFont.DrawString("DPAD_UP ");
            }

            if (state.DPad.Down == ButtonState.Pressed)
            {
                mFont.DrawString("DPAD_DOWN ");
            }
        }

        protected void drawThumbStick(Vector2 vec)
        {
            mFont.DrawString(vec.X + " / " + vec.Y);
        }

        protected void drawTriggers(GamePadState state)
        {
            mFont.DrawString(state.Triggers.Left + " / " + state.Triggers.Right);
        }
    }
}